#!/bin/bash
# -*- mode: sh; indent-tabs-mode: nil; sh-basic-offset: 4; -*-
# vim: et sts=4 sw=4

#  SPDX-License-Identifier: LGPL-2.1+
#
#  Copyright © 2019-2022 Collabora Ltd.
#  Copyright © 2019-2020 Valve Corporation.
#
#  This file is part of steamos-customizations.
#
#  steamos-customizations is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Lesser General Public License as
#  published by the Free Software Foundation; either version 2.1 of the License,
#  or (at your option) any later version.

set -e
set -u

#
# Helpers
#

log () { echo >&2 "$@"; }
fail() { echo >&2 "Error:" "$@"; exit 1; }

SYMLINKS_DIR=/dev/disk/by-partsets
RAUC_RUNTIME_DIR=/run/rauc
REBOOT_FOR_UPDATE=/run/steamos-atomupd/reboot_for_update


#
# Ensure that the UUID of the update is different from the current rootfs
#

# `steamos-image-recipes` is expected to include a UUID text file inside the
# RAUC bundle
target_uuid=
if [[ -f ${RAUC_BUNDLE_MOUNT_POINT}/UUID ]]; then
    target_uuid=$(<"${RAUC_BUNDLE_MOUNT_POINT}/UUID")
elif [[ -f ${RAUC_BUNDLE_MOUNT_POINT}/rootfs.img ]]; then
    target_uuid=$(blkid -o value -s UUID ${RAUC_BUNDLE_MOUNT_POINT}/rootfs.img)
fi

if [[ ${target_uuid} ]]; then
    this_image=$(steamos-bootconf this-image)
    root_uuid=$(blkid -o value -s UUID ${SYMLINKS_DIR}/${this_image}/rootfs)

    log "Current image rootfs UUID: $root_uuid"
    log "Target image rootfs UUID: $target_uuid"

    if [[ ${root_uuid} == ${target_uuid} ]]; then
        fail "Cannot install an image with the same rootfs UUID as the current active one"
    fi
else
    log "Trying to install an update that doesn't provide its UUID, trying to continue anyway"
fi


[ -d "$RAUC_RUNTIME_DIR" ] || fail "RAUC runtime dir does not exist"

#
# We are going to install a new update, remove the eventual info about a
# pending reboot
#

rm -f -- "$REBOOT_FOR_UPDATE"

#
# Disable all systemd timers
#

ACTIVE_TIMERS=$RAUC_RUNTIME_DIR/active-timers

systemctl list-units --full --no-legend --type=timer --state=active | \
while read -r service _; do
    [[ -n "$service" ]] || continue
    systemctl stop "$service" >/dev/null
    echo "$service"
done > "$ACTIVE_TIMERS"
