/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2024 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;

abstract class AbstractComm {

    private class TcpSocket {
        public Socket socket;
        public ObjectInputStream input;
        public ObjectOutputStream output;
    }

    TcpSocket socket = null;
    int port;

    protected abstract boolean processCommand(int command) throws IOException;
    protected abstract void cleanupResources();

    protected AbstractComm(int aport) {
        port = aport;
    }

    protected ObjectInputStream in() {
        return socket.input;
    }

    protected ObjectOutputStream out() {
        return socket.output;
    }

    public void listenAndAccept() throws IOException {
        ServerSocket srv_socket = null;
        Socket data_socket = null;

        srv_socket = new ServerSocket(port);
        data_socket = srv_socket.accept();
        srv_socket.close();

        commonConnect(data_socket);
    }

    public void connect(String addr) throws IOException {
        commonConnect(new Socket(InetAddress.getByName(addr), port));
    }

    private void commonConnect(Socket data_socket) throws IOException {

        data_socket.setTcpNoDelay(true);

        TcpSocket s = new TcpSocket();

        s.socket = data_socket;
        s.output = new ObjectOutputStream(data_socket.getOutputStream());
        s.input = new ObjectInputStream(data_socket.getInputStream());
        socket = s;
    }

    private void runServerLoop() throws IOException {
        if (null == socket)
            throw new IOException("Not connected");

        while (true) {
            int cmd;
            cmd = socket.input.readInt();
            if (cmd == 0xffffffff) {
                break;
            }
            if (false == processCommand(cmd)) {
                socket.input.skipBytes(socket.input.available());
                socket.output.writeInt(0);
                socket.output.flush();
            }
        }
    }

    public void runSession() throws IOException {
        runServerLoop();
        closeSession();
    }

    public void closeSession() throws IOException {
        if (null != socket) {
            socket.output.close();
            socket.input.close();
            socket.socket.close();
            socket = null;
        }
        cleanupResources();
        return;
    }
}
