/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2024 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

public class Blob implements Externalizable {
    public int id;
    public int[] args;
    public String[] strings;
    public byte[] data;

    private static final long serialVersionUID = 0x01;

    public Blob() {
        id = 0;
    }

    public Blob(int aId) {
        id = aId;
    }

    public String getString(int Index) {
        if (null == strings)
            return null;
        if (Index >= strings.length)
            return null;
        return strings[Index];
    }

    public long getLong(int Index) {
        if (null == args)
            return 0;
        if ((Index + 1) >= args.length)
            return 0;

        long v = args[Index];
        v = v << 32;
        v = v | args[Index + 1];

        return v;
    }

    public void writeExternal(ObjectOutput out) throws IOException {

        byte flags = 1;
        int maxbytes = 2;

        if (null != args) {
            if (args.length != 0) {
                flags |= 2;
                maxbytes += 4 * args.length;
            }
            if (maxbytes > 1000)
                throw new IOException();
        }

        if (null != strings) {
            if (strings.length != 0) {
                flags |= 4;
                for (int i = 0; i < strings.length; i++) {
                    if (strings[i] != null) {
                        maxbytes += strings[i].length() * 4;
                    }
                    maxbytes += 2;
                }
            }
        }

        if (maxbytes > 1000) {
            flags |= 16;
        }

        if (null != data) {
            if (data.length != 0) {
                flags |= 8;
                if ((data.length + maxbytes) > 1000) {
                    flags |= 32;
                }
            }
        }

        out.writeByte(flags);

        if ((flags & 2) != 0) {
            out.writeInt(args.length);
            for (int i = 0; i < args.length; i++) {
                out.writeInt(args[i]);
            }
        }

        if ((flags & 4) != 0) {
            out.writeInt(strings.length);
            if ((flags & 16) == 0) {
                for (int i = 0; i < strings.length; i++) {
                    if (strings[i] != null) {
                        out.writeUTF(strings[i]);
                    } else {
                        out.writeUTF("");
                    }
                }
            }
        }

        out.writeInt(id);

        if ((flags & 8) != 0) {
            out.writeInt(data.length);
            if ((flags & 32) == 0) {
                out.write(data);
            }
        }
        out.flush();

        if ((flags & 16) != 0) {
            for (int i = 0; i < strings.length; i++) {
                if (strings[i] != null) {
                    out.writeUTF(strings[i]);
                } else {
                    out.writeUTF("");
                }
                out.flush();
            }
        }

        if ((flags & 32) != 0) {
            out.write(data);
            out.flush();
        }

    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        byte flags = in.readByte();
        readObjectBody(in, flags);
    }

    private void readObjectBody(ObjectInput in, byte flags) throws IOException {
        if ((flags & 1) == 0)
            throw new IOException();

        if ((flags & 2) != 0) {
            args = new int[in.readInt()];
            for (int i = 0; i < args.length; i++) {
                args[i] = in.readInt();
            }
        } else {
            args = null;
        }

        if ((flags & 4) != 0) {
            strings = new String[in.readInt()];
            for (int i = 0; i < strings.length; i++) {
                strings[i] = in.readUTF();
            }
        } else {
            strings = null;
        }

        id = in.readInt();

        if ((flags & 8) != 0) {
            data = new byte[in.readInt()];
            in.readFully(data);
        } else {
            data = null;
        }
    }

    public static void writeBlob(ObjectOutput out, Blob blob) throws IOException {
        if (null == blob) {
            out.writeByte(0);
            out.flush();
        } else {
            blob.writeExternal(out);
        }
    }

    public static Blob readBlob(ObjectInput in) throws IOException {
        byte flags = in.readByte();
        if (flags == 0)
            return null;
        Blob blob = new Blob();
        blob.readObjectBody(in, flags);
        return blob;
    }

}
