// vim: set tabstop=4 shiftwidth=4 expandtab:
/*
Gwenview: an image viewer
Copyright 2008 Aurélien Gâteau <agateau@kde.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Cambridge, MA 02110-1301, USA.

*/
// Self
#include "moc_messageviewadapter.cpp"

// Katie
#include <QGraphicsWidget>
#include <QStyleOptionGraphicsItem>
#include <QPainter>
#include <QPen>

// KDE
#include <KLocale>
#include <KColorScheme>
#include <KDebug>

// Local
#include <lib/document/document.h>

namespace Gwenview
{

static const qreal s_roundness = 4;
static const qreal s_bordersize = 0.6;
static const qreal s_margin = 4;

class MessageViewWidget : public QGraphicsWidget
{
    Q_OBJECT
public:
    enum MessageType {
        InformationType,
        ErrorType
    };

    MessageViewWidget(MessageViewAdapter *adapter);

    void setMessage(const MessageType type, const QString &main, const QString &detail);

    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget = 0) final;
    void keyPressEvent(QKeyEvent *event) final;
    void changeEvent(QEvent *event) final;

private:
    void updateColors();

    MessageViewAdapter* mAdapter;
    MessageType mMessageType;
    QString mText;
    QTextOption mTextOptions;
    QColor mInfoBg;
    QColor mInfoBorder;
    QColor mErrorBg;
    QColor mErrorBorder;
};

MessageViewWidget::MessageViewWidget(MessageViewAdapter *adapter)
    : QGraphicsWidget(),
    mAdapter(adapter)
{
    mTextOptions.setAlignment(Qt::AlignCenter);
    mTextOptions.setWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);
    updateColors();
}

void MessageViewWidget::setMessage(const MessageType type, const QString &main, const QString &detail)
{
    mMessageType = type;
    mText = main;
    if (!detail.isEmpty()) {
        mText.append(QLatin1String(":\n"));
        mText.append(detail);
    }
    update();
}

void MessageViewWidget::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(widget);
    const QRectF boundingrect = boundingRect();
    QRectF mainrect = painter->boundingRect(boundingrect, mText, mTextOptions);
    mainrect = mainrect.adjusted(-s_margin, -s_margin, s_margin, s_margin);
    QBrush bgbrush;
    QPen borderpen;
    switch (mMessageType) {
        case MessageViewWidget::InformationType: {
            bgbrush = QBrush(mInfoBg);
            borderpen = QPen(mInfoBorder);
            break;
        }
        case MessageViewWidget::ErrorType: {
            bgbrush = QBrush(mErrorBg);
            borderpen = QPen(mErrorBorder);
            break;
        }
    }
    painter->setRenderHint(QPainter::Antialiasing);
    borderpen.setWidth(s_bordersize * 2);
    painter->setPen(borderpen);
    painter->drawRoundedRect(mainrect, s_roundness, s_roundness, Qt::AbsoluteSize);
    painter->setBrush(bgbrush);
    const QRectF subrect = mainrect.adjusted(s_bordersize, s_bordersize, -s_bordersize, -s_bordersize);
    painter->drawRoundedRect(subrect, s_roundness, s_roundness, Qt::AbsoluteSize);
    painter->setPen(option->palette.color(QPalette::Text));
    painter->drawText(mainrect, mText, mTextOptions);
}

void MessageViewWidget::updateColors()
{
    const KColorScheme scheme(QPalette::Active, KColorScheme::Window);
    mInfoBg = scheme.background(KColorScheme::PositiveBackground).color();
    mInfoBorder = KColorScheme::shade(mInfoBg, KColorScheme::DarkShade);
    mErrorBg = scheme.background(KColorScheme::NegativeBackground).color();
    mErrorBorder = KColorScheme::shade(mErrorBg, KColorScheme::DarkShade);
}

void MessageViewWidget::keyPressEvent(QKeyEvent *event)
{
    if (event->modifiers() != Qt::NoModifier) {
        QGraphicsWidget::keyPressEvent(event);
        return;
    }

    switch (event->key()) {
        case Qt::Key_Left:
        case Qt::Key_Up: {
            event->accept();
            mAdapter->previousImageRequested();
            return;
        }
        case Qt::Key_Right:
        case Qt::Key_Down: {
            event->accept();
            mAdapter->nextImageRequested();
            return;
        }
        default: {
            QGraphicsWidget::keyPressEvent(event);
            break;
        }
    }
}

void MessageViewWidget::changeEvent(QEvent *event)
{
    QGraphicsWidget::changeEvent(event);
    if (event->type() == QEvent::PaletteChange) {
        updateColors();
    }
}

struct MessageViewAdapterPrivate
{
    Document::Ptr mDocument;
    MessageViewWidget* mMessageViewWidget;
};

MessageViewAdapter::MessageViewAdapter()
: d(new MessageViewAdapterPrivate())
{
    d->mMessageViewWidget = new MessageViewWidget(this);
    setWidget(d->mMessageViewWidget);

    setInfoMessage(i18n("No document selected"));
}

MessageViewAdapter::~MessageViewAdapter()
{
    delete d;
}

void MessageViewAdapter::setErrorMessage(const QString& main, const QString& detail)
{
    if (main.isEmpty()) {
        d->mMessageViewWidget->hide();
        return;
    }
    d->mMessageViewWidget->show();
    d->mMessageViewWidget->setMessage(MessageViewWidget::ErrorType, main, detail);
}

void MessageViewAdapter::setInfoMessage(const QString& message)
{
    if (message.isEmpty()) {
        d->mMessageViewWidget->hide();
        return;
    }
    d->mMessageViewWidget->show();
    d->mMessageViewWidget->setMessage(MessageViewWidget::InformationType, message, QString());
}

Document::Ptr MessageViewAdapter::document() const
{
    return d->mDocument;
}

void MessageViewAdapter::setDocument(Document::Ptr doc)
{
    d->mDocument = doc;
}

} // namespace

#include "messageviewadapter.moc"

