/*
 *   Copyright (C) 2007 Ryan P. Bitanga <ephebiphobic@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "spellcheck.h"

#include <QClipboard>
#include <KApplication>
#include <KGlobal>
#include <KIcon>
#include <KDebug>

SpellCheckRunner::SpellCheckRunner(QObject* parent, const QVariantList &args)
    : Plasma::AbstractRunner(parent, args)
{
    KGlobal::locale()->insertCatalog(QLatin1String("krunner_spellcheckrunner"));
    setObjectName(QLatin1String("Spell Checker"));
    setIgnoredTypes(Plasma::RunnerContext::FileSystem | Plasma::RunnerContext::NetworkLocation);
    setSpeed(AbstractRunner::SlowSpeed);
}

void SpellCheckRunner::init()
{
    Plasma::AbstractRunner::init();
    m_speller = new KSpeller(KGlobal::config().data(), this);
}

void SpellCheckRunner::reloadConfiguration()
{
#warning TODO: untranslated keyword match
    m_triggerWord = config().readEntry("trigger", i18n("spell"));

    m_requireTriggerWord = config().readEntry("requireTriggerWord", true);

    Plasma::RunnerSyntax s(i18nc("Spelling checking runner syntax, first word is trigger word, e.g.  \"spell\".",
                                 "%1:q:", m_triggerWord),
                           i18n("Checks the spelling of :q:."));

    if (!m_requireTriggerWord) {
        s.addExampleQuery(QLatin1String(":q:"));
    }

    QList<Plasma::RunnerSyntax> syns;
    syns << s;
    setSyntaxes(syns);
}

void SpellCheckRunner::match(Plasma::RunnerContext &context)
{
    if (!context.isValid()) {
        return;
    }

    const QString term = context.query();
    QStringList terms = term.split(' ', QString::SkipEmptyParts);
    // qDebug() << Q_FUNC_INFO << m_requireTriggerWord << m_triggerWord << terms;
    if (m_requireTriggerWord) {
        if (terms.size() < 1 || terms.at(0) != m_triggerWord) {
            return;
        }
        terms.removeFirst();
    }

    QString spellerdictionary;
    if (terms.size() > 1 && m_speller->dictionaries().contains(terms.at(0))) {
        spellerdictionary = terms.at(0);
        terms.removeFirst();
    } else {
        spellerdictionary = KSpeller::defaultLanguage();
    }
    if (!m_speller->setDictionary(spellerdictionary)) {
        Plasma::QueryMatch match(this);
        match.setIcon(KIcon(QLatin1String("task-attention")));
        match.setText(i18n("Could not find a dictionary."));
        match.setEnabled(false);
        context.addMatch(match);
        return;
    }


    foreach (const QString &word, terms) {
        const bool correct = m_speller->check(word);
        if (correct) {
            Plasma::QueryMatch match(this);
            match.setIcon(KIcon(QLatin1String("checkbox")));
            match.setText(i18n("Correct: ") + word);
            match.setEnabled(false);
            context.addMatch(match);
            continue;
        }

        const QStringList suggestions = m_speller->suggest(word);
        if (suggestions.isEmpty()) {
            Plasma::QueryMatch match(this);
            match.setIcon(KIcon(QLatin1String("edit-delete")));
            match.setText(i18n("Incorrect but no suggestions for: %1", word));
            match.setEnabled(false);
            context.addMatch(match);
            continue;
        }

        foreach (const QString &suggestion, suggestions) {
            Plasma::QueryMatch match(this);
            match.setIcon(KIcon(QLatin1String("edit-delete")));
            match.setText(i18n("Suggested words: %1", suggestion));
            match.setData(suggestion);
            context.addMatch(match);
        }
    }
}

void SpellCheckRunner::run(const Plasma::QueryMatch &match)
{
    // Copy words to clipboard
    kapp->clipboard()->setText(match.data().toString());
}

#include "moc_spellcheck.cpp"
