/*
 *   Copyright 2008 Aaron Seigo <aseigo@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef PLASMA_LABEL_H
#define PLASMA_LABEL_H

#include <QtGui/QGraphicsProxyWidget>

#include <plasma/plasma_export.h>

#include <QLabel>

namespace Plasma
{

class LabelPrivate;

/**
 * @class Label plasma/widgets/label.h <Plasma/Widgets/Label>
 *
 * @short Provides a plasma-themed QLabel.
 */
class PLASMA_EXPORT Label : public QGraphicsProxyWidget
{
    Q_OBJECT
    Q_PROPERTY(QString text READ text WRITE setText)
    Q_PROPERTY(Qt::Alignment alignment READ alignment WRITE setAlignment)
    Q_PROPERTY(bool hasScaledContents READ hasScaledContents WRITE setScaledContents)
    Q_PROPERTY(bool wordWrap READ wordWrap WRITE setWordWrap)
    Q_PROPERTY(bool elideText READ elideText WRITE setElideText)
public:
    /**
     * Constructs a label with word wrap on by default
     *
     * @param parent the parent of this widget
     */
    explicit Label(QGraphicsWidget *parent = nullptr);
    ~Label();

    /**
     * Sets the display text for this Label
     *
     * @param text the text to display; should be translated.
     */
    void setText(const QString &text);

    /**
     * @return the display text
     */
    QString text() const;

    /**
     * Sets the alignment for the text
     *
     * @param the desired alignment
     */
    void setAlignment(Qt::Alignment alignment);

    /**
     * @return the alignment for the text used in the labels
     */
    Qt::Alignment alignment() const;

    /**
     * Scale or not the contents of the label to the label size
     *
     * @param scale
     */
    void setScaledContents(bool scaled);

    /**
     * @return true if the contents are scaled to the label size
     */
    bool hasScaledContents() const;

    /**
     * Set if the text on the label can be selected with the mouse
     *
     * @param enable true if we want to manage text selection with the mouse
     * @since 4.4 
     */
    void setTextSelectable(bool enable);
    
    /**
     * @return true if the text is selectable with the mouse
     * @since 4.4
     */
    bool textSelectable() const;

    /**
     * Sets if the text of the label can wrap in multiple lines
     *
     * @param wrap multiple lines or not
     * @since 4.5
     */
    void setWordWrap(bool wrap);

    /**
     * @return true if the label text can wrap in multiple lines if too long
     * @since 4.5
     */
    bool wordWrap() const;

    /**
     * Sets if the text of the label can elide if too long
     *
     * @param elide elide text or not
     * @note this makes sense only when not wrapping, turn wrapping off for best results
     * @since 4.24
     */
    void setElideText(bool elide);

    /**
     * @return true if the label text can elide if too long
     * @since 4.24
     */
    bool elideText() const;

    /**
     * @return the native widget wrapped by this Label
     */
    QLabel *nativeWidget() const;

Q_SIGNALS:
    void linkActivated(const QString &link);
    void linkHovered(const QString &link);

protected:
    void resizeEvent(QGraphicsSceneResizeEvent *event);
    void changeEvent(QEvent *event);
    bool event(QEvent *event);
    QVariant itemChange(GraphicsItemChange change, const QVariant & value);
    void contextMenuEvent(QGraphicsSceneContextMenuEvent *event);
    QSizeF sizeHint(Qt::SizeHint which, const QSizeF &constraint) const;

private:
    Q_PRIVATE_SLOT(d, void setPalette())

    LabelPrivate * const d;
};

} // namespace Plasma

#endif // multiple inclusion guard
