/*  This file is part of the KDE libraries
    Copyright (C) 2022 Ivailo Monev <xakepa10@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2, as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "klockfile.h"
#include "kglobal.h"
#include "kstandarddirs.h"
#include "kdebug.h"
#include "kde_file.h"

#include <QStandardPaths>
#include <QCoreApplication>
#include <QThread>

#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>

#define KLOCKFILE_TIMEOUT 150
#define KLOCKFILE_SLEEPTIME 50

class KLockFilePrivate
{
public:
    KLockFilePrivate();

    QString m_lockname;
    QByteArray m_lockfile;
    int m_lockfd;
};

KLockFilePrivate::KLockFilePrivate()
    : m_lockfd(-1)
{
}

KLockFile::KLockFile(const QString &file)
    : d(new KLockFilePrivate())
{
    d->m_lockname = file;
    // NOTE: KConfig may attempt to create KLockFile from its destructor when KGlobal is no more
    // thus QStandardPaths::writableLocation() is used here
    d->m_lockfile = QFile::encodeName(QStandardPaths::writableLocation(QStandardPaths::RuntimeLocation));
    d->m_lockfile.append('/');
    d->m_lockfile.append(QByteArray::number(qHash(file)));
    d->m_lockfile.append(".klockfile");
}

KLockFile::~KLockFile()
{
    unlock();
    d->m_lockfile.clear();
    delete d;
}

bool KLockFile::tryLock()
{
    if (d->m_lockfd != -1) {
        return true;
    }

    d->m_lockfd = KDE_open(d->m_lockfile.constData(), O_WRONLY | O_CREAT | O_EXCL, 0644);
    return (d->m_lockfd != -1);
}

void KLockFile::lock()
{
    kDebug() << "locking" << d->m_lockname << d->m_lockfile;
    while (!tryLock() && !d->m_lockfile.isEmpty()) {
        QCoreApplication::processEvents(QEventLoop::AllEvents, KLOCKFILE_TIMEOUT);
        QThread::msleep(KLOCKFILE_SLEEPTIME);
    }
    kDebug() << "locked" << d->m_lockname << d->m_lockfile;
}

bool KLockFile::isLocked() const
{
    return (d->m_lockfd != -1);
}

void KLockFile::unlock()
{
    if (d->m_lockfd != -1) {
        QT_CLOSE(d->m_lockfd);
        if (Q_UNLIKELY(::unlink(d->m_lockfile.constData()) == -1)) {
            const int savederrno = errno;
            kWarning() << "could not remove lock file" << qt_error_string(savederrno);
        }
        d->m_lockfd = -1;
        kDebug() << "unlocked" << d->m_lockname << d->m_lockfile;
    }
}
