/*  This file is part of the KDE libraries
 *  Copyright 2007, 2010 David Faure <faure@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef KMIMEGLOBSFILEPARSER_P_H
#define KMIMEGLOBSFILEPARSER_P_H

#include <kdebug.h>
#include <QtCore/QHash>

#include <QIODevice>
#include <QString>
#include <QStringList>

/**
 * @internal
 */
class KMimeGlobsFileParser
{
public:
    typedef QHash<QString, QStringList> PatternsMap; // mimetype -> patterns

    struct Glob {
        Glob()
            : weight(50), casesensitive(Qt::CaseInsensitive)
        {
        }

        Glob(const QString &mime, int w, const QString &pat, bool cs)
            : weight(w), casesensitive(cs ? Qt::CaseSensitive : Qt::CaseInsensitive), pattern(pat), mimeType(mime)
        {
        }

        int weight;
        Qt::CaseSensitivity casesensitive;
        QString pattern;
        QString mimeType;
    };

    class GlobList : public QList<Glob>
    {
    public:
        bool hasPattern(const QString &mime, const QString &pattern) const
        {
            const_iterator it = begin();
            const const_iterator myend = end();
            for (; it != myend; ++it)
                if ((*it).pattern == pattern && (*it).mimeType == mime)
                    return true;
            return false;
        }
        // "noglobs" is very rare occurrence, so it's ok if it's slow
        void removeMime(const QString &mime)
        {
            QMutableListIterator<Glob> it(*this);
            while (it.hasNext()) {
                if (it.next().mimeType == mime)
                    it.remove();
            }
        }

        // for tests
        PatternsMap patternsMap() const
        {
            PatternsMap patMap;
            patMap.reserve(this->size());
            const_iterator it = begin();
            const const_iterator myend = end();
            for (; it != myend; ++it)
                patMap[(*it).mimeType].append((*it).pattern);
            return patMap;
        }
    };

    // Read globs (patterns) files
    static GlobList parseGlobs(const QStringList &globFiles);

    static bool parseGlobFile(QIODevice *file, GlobList &globs);
};

#endif /* KMIMEFILEPARSER_H */

