/* This file is part of the KDE libraries
   Copyright (C) 1999 Torben Weis <weis@kde.org>
   Copyright (C) 2003 Waldo Bastian <bastian@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "kprotocolinfo.h"
#include "kprotocolinfo_p.h"
#include "kprotocolinfofactory.h"

#include <kmimetypetrader.h>
#include <kstandarddirs.h>
#include <kconfig.h>
#include <kconfiggroup.h>

static void selectServiceOrHelper(const QString &protocol, KProtocolInfo::Ptr &returnProtocol, KService::Ptr &returnService)
{
    // have up to two sources of data:
    // 1) the exec line of the .protocol file, if there's one (could be a kioslave or a helper app)
    // 2) the application associated with x-scheme-handler/<protocol> if there's one

    // if both exist, then:
    //  A) if the .protocol file says "launch an application", then the new-style handler-app has priority
    //  B) but if the .protocol file is for a kioslave (e.g. kio_http) then this has priority over
    //     firefox or chromium saying x-scheme-handler/http. Gnome people want to send all HTTP urls
    //     to a webbrowser, but mimetype-determination-in-calling-application by default is done here

    const KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (prot) {
        returnProtocol = prot;
        return;
    }
    // no protocol file, use handler app if any
    returnService = KMimeTypeTrader::self()->preferredService(QString::fromLatin1("x-scheme-handler/") + protocol);
}

KProtocolInfo::KProtocolInfo(const QString &path)
    : KSycocaEntry(*new KProtocolInfoPrivate(path, this))
{
    Q_D(KProtocolInfo);

    KConfig sconfig(KStandardDirs::locate("services", path));
    KConfigGroup config(&sconfig, "Protocol");

    m_name = config.readEntry("protocol");
    m_exec = config.readPathEntry("exec", QString());
    m_isSourceProtocol = config.readEntry("source", true);
    m_supportsListing = config.readEntry("listing", false);
    m_supportsReading = config.readEntry("reading", false);
    m_supportsWriting = config.readEntry("writing", false);
    m_supportsMakeDir = config.readEntry("makedir", false);
    m_supportsDeleting = config.readEntry("deleting", false);
    m_supportsLinking = config.readEntry("linking", false);
    m_supportsMoving = config.readEntry("moving", false);
    m_canCopyFromFile = config.readEntry("copyFromFile", false);
    m_canCopyToFile = config.readEntry("copyToFile", false);
    m_defaultMimetype = config.readEntry("defaultMimetype");
    m_determineMimetypeFromExtension = config.readEntry("determineMimetypeFromExtension", true);
    m_icon = config.readEntry("Icon", QString());
    m_config = config.readEntry("config", m_name);
    m_maxSlaves = config.readEntry("maxInstances", 1);

    d->canRenameFromFile = config.readEntry("renameFromFile", false);
    d->canRenameToFile = config.readEntry("renameToFile", false);
    d->canDeleteRecursive = config.readEntry("deleteRecursive", false);
    const QString fnu = config.readEntry( "fileNameUsedForCopying", "FromURL");
    d->fileNameUsedForCopying = FromUrl;
    if (fnu == QLatin1String("Name")) {
        d->fileNameUsedForCopying = Name;
    } else if (fnu == QLatin1String("DisplayName")) {
        d->fileNameUsedForCopying = DisplayName;
    }
    d->maxSlavesPerHost = config.readEntry("maxInstancesPerHost", 0);
    d->docPath = config.readPathEntry("X-DocPath", QString());
    d->local = config.readEntry("local", false);
    d->showPreviews = config.readEntry("ShowPreviews", d->local);
}

KProtocolInfo::KProtocolInfo(QDataStream &str, int offset)
    : KSycocaEntry(*new KProtocolInfoPrivate(str, offset, this))
{
    load(str);
}

KProtocolInfo::~KProtocolInfo()
{
}

void KProtocolInfo::load(QDataStream &str)
{
    Q_D(KProtocolInfo);
    // NOTE: make sure to update the version number in ksycoca_p.h
    qint8 i_isSourceProtocol, i_supportsListing,
          i_supportsReading, i_supportsWriting,
          i_supportsMakeDir, i_supportsDeleting,
          i_supportsLinking, i_supportsMoving,
          i_determineMimetypeFromExtension, i_canCopyFromFile,
          i_canCopyToFile, i_showPreviews,
          i_canRenameFromFile, i_canRenameToFile,
          i_canDeleteRecursive, i_fileNameUsedForCopying;

    str >> m_name >> m_exec >> m_defaultMimetype
        >> i_determineMimetypeFromExtension
        >> m_icon
        >> i_isSourceProtocol >> i_supportsListing
        >> i_supportsReading >> i_supportsWriting
        >> i_supportsMakeDir >> i_supportsDeleting
        >> i_supportsLinking >> i_supportsMoving
        >> i_canCopyFromFile >> i_canCopyToFile
        >> m_config >> m_maxSlaves >> d->docPath >> d->local
        >> i_showPreviews
        >> i_canRenameFromFile >> i_canRenameToFile
        >> i_canDeleteRecursive >> i_fileNameUsedForCopying
        >> d->maxSlavesPerHost;

    m_isSourceProtocol = (i_isSourceProtocol != 0);
    m_supportsListing = (i_supportsListing != 0);
    m_supportsReading = (i_supportsReading != 0);
    m_supportsWriting = (i_supportsWriting != 0);
    m_supportsMakeDir = (i_supportsMakeDir != 0);
    m_supportsDeleting = (i_supportsDeleting != 0);
    m_supportsLinking = (i_supportsLinking != 0);
    m_supportsMoving = (i_supportsMoving != 0);
    m_canCopyFromFile = (i_canCopyFromFile != 0);
    m_canCopyToFile = (i_canCopyToFile != 0);
    d->canRenameFromFile = (i_canRenameFromFile != 0);
    d->canRenameToFile = (i_canRenameToFile != 0);
    d->canDeleteRecursive = (i_canDeleteRecursive != 0);
    d->fileNameUsedForCopying = FileNameUsedForCopying(i_fileNameUsedForCopying);
    m_determineMimetypeFromExtension = (i_determineMimetypeFromExtension != 0);
    d->showPreviews = (i_showPreviews != 0);
}

void KProtocolInfoPrivate::save(QDataStream &str)
{
    KSycocaEntryPrivate::save(str);

    // NOTE: make sure to update the version number in ksycoca_p.h
    qint8 i_isSourceProtocol, i_supportsListing,
          i_supportsReading, i_supportsWriting,
          i_supportsMakeDir, i_supportsDeleting,
          i_supportsLinking, i_supportsMoving,
          i_determineMimetypeFromExtension, i_canCopyFromFile,
          i_canCopyToFile, i_showPreviews,
          i_canRenameFromFile, i_canRenameToFile,
          i_canDeleteRecursive, i_fileNameUsedForCopying;

    i_isSourceProtocol = q->m_isSourceProtocol ? 1 : 0;
    i_supportsListing = q->m_supportsListing ? 1 : 0;
    i_supportsReading = q->m_supportsReading ? 1 : 0;
    i_supportsWriting = q->m_supportsWriting ? 1 : 0;
    i_supportsMakeDir = q->m_supportsMakeDir ? 1 : 0;
    i_supportsDeleting = q->m_supportsDeleting ? 1 : 0;
    i_supportsLinking = q->m_supportsLinking ? 1 : 0;
    i_supportsMoving = q->m_supportsMoving ? 1 : 0;
    i_canCopyFromFile = q->m_canCopyFromFile ? 1 : 0;
    i_canCopyToFile = q->m_canCopyToFile ? 1 : 0;
    i_canRenameFromFile = canRenameFromFile ? 1 : 0;
    i_canRenameToFile = canRenameToFile ? 1 : 0;
    i_canDeleteRecursive = canDeleteRecursive ? 1 : 0;
    i_fileNameUsedForCopying = int(fileNameUsedForCopying);
    i_determineMimetypeFromExtension = q->m_determineMimetypeFromExtension ? 1 : 0;
    i_showPreviews = showPreviews ? 1 : 0;

    str << q->m_name << q->m_exec << q->m_defaultMimetype
        << i_determineMimetypeFromExtension
        << q->m_icon
        << i_isSourceProtocol << i_supportsListing
        << i_supportsReading << i_supportsWriting
        << i_supportsMakeDir << i_supportsDeleting
        << i_supportsLinking << i_supportsMoving
        << i_canCopyFromFile << i_canCopyToFile
        << q->m_config << q->m_maxSlaves << docPath << local
        << i_showPreviews
        << i_canRenameFromFile << i_canRenameToFile
        << i_canDeleteRecursive << i_fileNameUsedForCopying
        << maxSlavesPerHost;
}

//
// Static functions:
//
QStringList KProtocolInfo::protocols()
{
    return KProtocolInfoFactory::self()->protocols();
}

QString KProtocolInfo::icon(const QString &protocol)
{
    KProtocolInfo::Ptr prot;
    KService::Ptr service;
    selectServiceOrHelper(protocol, prot, service);
    if (service) {
        return service->icon();
    } else if (prot) {
        return prot->m_icon;
    }
    return QString();
}

QString KProtocolInfo::config(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (!prot) {
        return QString();
    }
    return QString::fromLatin1("kio_%1rc").arg(prot->m_config);
}

int KProtocolInfo::maxSlaves(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (!prot) {
        return 1;
    }
    return prot->m_maxSlaves;
}

int KProtocolInfo::maxSlavesPerHost(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (!prot) {
        return 0;
    }
    return prot->d_func()->maxSlavesPerHost;
}

bool KProtocolInfo::determineMimetypeFromExtension(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (!prot) {
        return true;
    }
    return prot->m_determineMimetypeFromExtension;
}

QString KProtocolInfo::exec(const QString &protocol)
{
    KProtocolInfo::Ptr prot;
    KService::Ptr service;
    selectServiceOrHelper(protocol, prot, service);
    if (service) {
        return service->exec();
    } else if (prot) {
        return prot->m_exec;
    }
    return QString();
}

QString KProtocolInfo::docPath(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (!prot) {
        return QString();
    }
    return prot->d_func()->docPath;
}

bool KProtocolInfo::protocolIsLocal(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (!prot) {
        return false;
    }
    return prot->d_func()->local;
}

bool KProtocolInfo::showFilePreview(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (!prot) {
        return false;
    }
    return prot->d_func()->showPreviews;
}

QString KProtocolInfo::defaultMimeType() const
{
    return m_defaultMimetype;
}

bool KProtocolInfo::supportsListing() const
{
    return m_supportsListing;
}

bool KProtocolInfo::canRenameFromFile() const
{
    Q_D(const KProtocolInfo);
    return d->canRenameFromFile;
}

bool KProtocolInfo::canRenameToFile() const
{
    Q_D(const KProtocolInfo);
    return d->canRenameToFile;
}

bool KProtocolInfo::canDeleteRecursive() const
{
    Q_D(const KProtocolInfo);
    return d->canDeleteRecursive;
}

KProtocolInfo::FileNameUsedForCopying KProtocolInfo::fileNameUsedForCopying() const
{
    Q_D(const KProtocolInfo);
    return d->fileNameUsedForCopying;
}

bool KProtocolInfo::isHelperProtocol(const KUrl &url)
{
    return isHelperProtocol(url.protocol());
}

bool KProtocolInfo::isHelperProtocol(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    if (prot) {
        return false;
    }
    const KService::Ptr service = KMimeTypeTrader::self()->preferredService(QString::fromLatin1("x-scheme-handler/") + protocol);
    return !service.isNull();
}

bool KProtocolInfo::isKnownProtocol(const KUrl &url)
{
    return isKnownProtocol(url.protocol());
}

bool KProtocolInfo::isKnownProtocol(const QString &protocol)
{
    KProtocolInfo::Ptr prot = KProtocolInfoFactory::self()->findProtocol(protocol);
    return prot || isHelperProtocol(protocol);
}
