/*
    This file is part of the KDE libraries
    Copyright (C) 2024 Ivailo Monev <xakepa10@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2, as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "kplasmajobtracker.h"

#include <QDBusInterface>
#include <QDBusConnectionInterface>
#include <kglobal.h>
#include <kcomponentdata.h>
#include <kaboutdata.h>
#include <kjob.h>
#include <kdebug.h>

static QString kJobID(const KJob *job)
{
    return QString::number(quintptr(job), 16);
}

class KPlasmaJobTrackerPrivate
{
public:
    KPlasmaJobTrackerPrivate();

    void _k_slotStopRequested(const QString &name);

    QMap<KJob*, QVariantMap> jobs;
    QDBusInterface desktop;
    QDBusInterface windowed;
};

KPlasmaJobTrackerPrivate::KPlasmaJobTrackerPrivate()
    : desktop("org.kde.plasma-desktop", "/JobTracker", "org.kde.JobTracker", QDBusConnection::sessionBus()),
    windowed("org.kde.plasma-windowed", "/JobTracker", "org.kde.JobTracker", QDBusConnection::sessionBus())
{
}

void KPlasmaJobTrackerPrivate::_k_slotStopRequested(const QString &name)
{
    kDebug() << "job stop requested" << name;
    foreach (KJob* job, jobs.keys()) {
        const QString jobid = kJobID(job);
        if (jobid == name) {
            job->kill(KJob::EmitResult);
            break;
        }
    }
}


KPlasmaJobTracker::KPlasmaJobTracker(QObject *parent)
    : KJobTrackerInterface(parent),
    d(new KPlasmaJobTrackerPrivate())
{
    if (d->desktop.isValid()) {
        connect(
            &d->desktop, SIGNAL(stopRequested(QString)),
            this, SLOT(_k_slotStopRequested(QString))
        );
    }
    if (d->windowed.isValid()) {
        connect(
            &d->windowed, SIGNAL(stopRequested(QString)),
            this, SLOT(_k_slotStopRequested(QString))
        );
    }
}

KPlasmaJobTracker::~KPlasmaJobTracker()
{
    const int registeredjobs = d->jobs.size();
    if (registeredjobs > 0) {
        // force-unregister any jobs in applets otherwise the interface for the jobs becomes
        // non-operational (there would be no jobs to stop anyway, this is the proxy to do it too)
        kWarning() << "there are" << registeredjobs << "registered jobs still";
        QMutableMapIterator<KJob*, QVariantMap> iter(d->jobs);
        while (iter.hasNext()) {
            iter.next();
            unregisterJob(iter.key());
            iter.remove();
        }
    }
    delete d;
}

bool KPlasmaJobTracker::registerJob(KJob *job)
{
    if (d->jobs.contains(job)) {
        kWarning() << "atempting to register the same job twice" << job;
        return false;
    }

    if (!d->desktop.isValid() && !d->windowed.isValid()) {
        kDebug() << "plasma job tracker not registered";
        return false;
    }

    const KComponentData componentData = KGlobal::mainComponent();
    QString appIconName = componentData.aboutData()->programIconName();
    if (appIconName.isEmpty()) {
        appIconName = componentData.aboutData()->appName();
    }

    const QString jobid = kJobID(job);
    QVariantMap jobdata;
    jobdata.insert("infoMessage", QString());
    jobdata.insert("appIconName", appIconName);
    jobdata.insert("labelName0", QString());
    jobdata.insert("labelName1", QString());
    jobdata.insert("label0", QString());
    jobdata.insert("label1", QString());
    // NOTE: destUrl never changes, it is set when the job is created
    jobdata.insert("destUrl", job->property("destUrl").toString());
    jobdata.insert("error", QString());
    jobdata.insert("percentage", 0);
    jobdata.insert("state", "running");
    jobdata.insert("killable", bool(job->capabilities() & KJob::Killable));
    d->jobs.insert(job, jobdata);
    if (d->desktop.isValid()) {
        d->desktop.call("addJob", jobid);
        d->desktop.asyncCall("updateJob", jobid, jobdata);
    }
    if (d->windowed.isValid()) {
        d->windowed.call("addJob", jobid);
        d->windowed.asyncCall("updateJob", jobid, jobdata);
    }
    kDebug() << "registerd job" << jobid << jobdata;
    return KJobTrackerInterface::registerJob(job);
}

void KPlasmaJobTracker::unregisterJob(KJob *job)
{
    KJobTrackerInterface::unregisterJob(job);

    if (!d->jobs.contains(job)) {
        return;
    }

    // both finished() and unregistrJob() will be called, either does it
    kDebug() << "unregisterd job" << kJobID(job);
    finished(job);
}

void KPlasmaJobTracker::finished(KJob *job)
{
    if (!d->jobs.contains(job)) {
        return;
    }

    const QString jobid = kJobID(job);
    QVariantMap jobdata = d->jobs.value(job);
    if (job->error() != KJob::NoError) {
        jobdata.insert("error", job->errorText());
    }
    jobdata.insert("state", "stopped");
    d->desktop.asyncCall("updateJob", jobid, jobdata);
    d->windowed.asyncCall("updateJob", jobid, jobdata);
    kDebug() << "job finished" << jobid;
    d->jobs.remove(job);
}

void KPlasmaJobTracker::suspended(KJob *job)
{
    if (!d->jobs.contains(job)) {
        return;
    }

    const QString jobid = kJobID(job);
    QVariantMap jobdata = d->jobs.value(job);
    jobdata.insert("state", "suspended");
    d->jobs.insert(job, jobdata);
    d->desktop.asyncCall("updateJob", jobid, jobdata);
    d->windowed.asyncCall("updateJob", jobid, jobdata);
    kDebug() << "job suspended" << jobid;
}

void KPlasmaJobTracker::resumed(KJob *job)
{
    if (!d->jobs.contains(job)) {
        return;
    }

    const QString jobid = kJobID(job);
    QVariantMap jobdata = d->jobs.value(job);
    jobdata.insert("state", "running");
    d->jobs.insert(job, jobdata);
    d->desktop.asyncCall("updateJob", jobid, jobdata);
    d->windowed.asyncCall("updateJob", jobid, jobdata);
    kDebug() << "job resumed" << jobid;
}

void KPlasmaJobTracker::description(KJob *job, const QString &title,
                                    const QPair<QString, QString> &field1,
                                    const QPair<QString, QString> &field2)
{
    if (!d->jobs.contains(job)) {
        return;
    }

    const QString jobid = kJobID(job);
    QVariantMap jobdata = d->jobs.value(job);
    jobdata.insert("labelName0", field1.first);
    jobdata.insert("label0", field1.second);
    jobdata.insert("labelName1", field2.first);
    jobdata.insert("label1", field2.second);
    d->jobs.insert(job, jobdata);
    d->desktop.asyncCall("updateJob", jobid, jobdata);
    d->windowed.asyncCall("updateJob", jobid, jobdata);
    kDebug() << "job description" << jobid << field1 << field2;
}

void KPlasmaJobTracker::infoMessage(KJob *job, const QString &plain, const QString &rich)
{
    if (!d->jobs.contains(job)) {
        return;
    }

    const QString jobid = kJobID(job);
    QVariantMap jobdata = d->jobs.value(job);
    jobdata.insert("infoMessage", plain);
    // NOTE: the message is used in the notificatin plasma applet, it should be stored
    d->jobs.insert(job, jobdata);
    d->desktop.asyncCall("updateJob", jobid, jobdata);
    d->windowed.asyncCall("updateJob", jobid, jobdata);
    kDebug() << "job info message" << jobid << plain << rich;
}

void KPlasmaJobTracker::percent(KJob *job, unsigned long percent)
{
    if (!d->jobs.contains(job)) {
        return;
    }

    const QString jobid = kJobID(job);
    QVariantMap jobdata = d->jobs.value(job);
    jobdata.insert("percentage", qulonglong(percent));
    d->jobs.insert(job, jobdata);
    d->desktop.asyncCall("updateJob", jobid, jobdata);
    d->windowed.asyncCall("updateJob", jobid, jobdata);
    kDebug() << "job percent" << jobid << percent;
}

#include "moc_kplasmajobtracker.cpp"
