/* This file is part of the KDE libraries
    Copyright (C) 2000 Stephan Kulow <coolo@kde.org>
                       David Faure <faure@kde.org>
    Copyright (C) 2006 Kevin Ottens <ervin@kde.org>
    Copyright (C) 2013 Dawit Alemayehu <adawit@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "jobuidelegate.h"
#include "kio/job.h"
#include "kjob.h"
#include "klocale.h"
#include "kmessagebox.h"
#include "ksharedconfig.h"
#include "kdebug.h"

#include <QPointer>
#include <QWidget>

KIO::RenameDialog_Result KIO::JobUiDelegate::askFileRename(KJob * job,
                                                           const QString & caption,
                                                           const QString& src,
                                                           const QString & dest,
                                                           KIO::RenameDialog_Mode mode,
                                                           QString& newDest,
                                                           KIO::filesize_t sizeSrc,
                                                           KIO::filesize_t sizeDest,
                                                           time_t ctimeSrc,
                                                           time_t ctimeDest,
                                                           time_t mtimeSrc,
                                                           time_t mtimeDest)
{
    Q_UNUSED(job);
    //kDebug() << "job=" << job;
    // We now do it in process, so that opening the rename dialog
    // doesn't start uiserver for nothing if progressId=0 (e.g. F2 in konq)
    KIO::RenameDialog dlg( window(), caption, src, dest, mode,
                                                     sizeSrc, sizeDest,
                                                     ctimeSrc, ctimeDest, mtimeSrc,
                                                     mtimeDest);
    dlg.setWindowModality(Qt::WindowModal);
    connect(job, SIGNAL(finished(KJob*)), &dlg, SLOT(reject())); // #192976
    KIO::RenameDialog_Result res = static_cast<RenameDialog_Result>(dlg.exec());
    if (res == R_AUTO_RENAME) {
        newDest = dlg.autoDestUrl().path();
    }
    else {
        newDest = dlg.newDestUrl().path();
    }
    return res;
}

KIO::SkipDialog_Result KIO::JobUiDelegate::askSkip(KJob *job,
                                              bool multi,
                                              const QString & error_text)
{
    // We now do it in process. So this method is a useless wrapper around KIO::open_RenameDialog.
    KIO::SkipDialog dlg( window(), multi, error_text );
    dlg.setWindowModality(Qt::WindowModal);
    connect(job, SIGNAL(finished(KJob*)), &dlg, SLOT(reject())); // #192976
    return static_cast<KIO::SkipDialog_Result>(dlg.exec());
}

bool KIO::JobUiDelegate::askDeleteConfirmation(const KUrl::List& urls,
                                               DeletionType deletionType,
                                               ConfirmationType confirmationType)
{
    QString keyName;
    bool ask = ( confirmationType == ForceConfirmation );
    if (!ask) {
        KSharedConfigPtr kioConfig = KSharedConfig::openConfig("kiorc", KConfig::NoGlobals);

        switch (deletionType ) {
        case Delete:
            keyName = "ConfirmDelete" ;
            break;
        case Trash:
            keyName = "ConfirmTrash" ;
            break;
        case EmptyTrash:
            keyName = "ConfirmEmptyTrash" ;
            break;
        }

        // The default value for confirmations is true (for both delete and trash)
        // If you change this, update kdebase/apps/konqueror/settings/konq/behaviour.cpp
        const bool defaultValue = true;
        ask = kioConfig->group("Confirmations").readEntry(keyName, defaultValue);
    }
    if (ask) {
        QStringList prettyList;
        Q_FOREACH(const KUrl& url, urls) {
            if ( url.protocol() == "trash" ) {
                QString path = url.path();
                // HACK (#98983): remove "0-foo". Note that it works better than
                // displaying KFileItem::name(), for files under a subdir.
                path.remove(QRegExp("^/[0-9]*-"));
                prettyList.append(path);
            } else {
                prettyList.append(url.pathOrUrl());
            }
        }

        int result;
        QWidget* widget = window();
        const KMessageBox::Options options = KMessageBox::Notify | KMessageBox::WindowModal;
        switch(deletionType) {
        case Delete:
            result = KMessageBox::warningContinueCancelList(
            widget,
            i18np("Do you really want to delete this item?", "Do you really want to delete these %1 items?", prettyList.count()),
            prettyList,
            i18n("Delete Files"),
            KStandardGuiItem::del(),
            KStandardGuiItem::cancel(),
            keyName, options);
            break;
        case EmptyTrash:
            result = KMessageBox::warningContinueCancel(
            widget,
            i18nc("@info", "Do you want to permanently delete all items from Trash? This action cannot be undone."),
            QString(),
            KGuiItem(i18nc("@action:button", "Empty Trash"),
            KIcon("user-trash")),
            KStandardGuiItem::cancel(),
            keyName, options);
            break;
        case Trash:
        default:
            result = KMessageBox::warningContinueCancelList(
                widget,
                i18np("Do you really want to move this item to the trash?", "Do you really want to move these %1 items to the trash?", prettyList.count()),
                prettyList,
                i18n("Move to Trash"),
                KGuiItem(i18nc("Verb", "&Trash"), "user-trash"),
                KStandardGuiItem::cancel(),
                keyName, options);
        }
        if (!keyName.isEmpty()) {
            // Check kmessagebox setting... erase & copy to konquerorrc.
            KSharedConfig::Ptr config = KGlobal::config();
            KConfigGroup notificationGroup(config, "Notification Messages");
            if (!notificationGroup.readEntry(keyName, true)) {
                notificationGroup.writeEntry(keyName, true);
                notificationGroup.sync();

                KSharedConfigPtr kioConfig = KSharedConfig::openConfig("kiorc", KConfig::NoGlobals);
                kioConfig->group("Confirmations").writeEntry(keyName, false);
            }
        }
        return (result == KMessageBox::Continue);
    }
    return true;
}

int KIO::JobUiDelegate::requestMessageBox(KIO::JobUiDelegate::MessageBoxType type,
                                          const QString& text, const QString& caption,
                                          const QString& buttonYes, const QString& buttonNo,
                                          const QString& iconYes, const QString& iconNo,
                                          const QString& dontAskAgainName)
{
    int result = -1;

    //kDebug() << type << text << "caption=" << caption;

    KConfig config("kioslaverc");
    KMessageBox::setDontShowAskAgainConfig(&config);

    const KGuiItem buttonYesGui (buttonYes, iconYes);
    const KGuiItem buttonNoGui (buttonNo, iconNo);
    KMessageBox::Options options = (KMessageBox::Notify | KMessageBox::WindowModal | KMessageBox::AllowLink);
    switch (type) {
        case KIO::JobUiDelegate::QuestionYesNo: {
            result = KMessageBox::questionYesNo(
                window(), text, caption, buttonYesGui,
                buttonNoGui, dontAskAgainName, options
            );
            break;
        }
        case KIO::JobUiDelegate::WarningYesNo:
            result = KMessageBox::warningYesNo(
                window(), text, caption, buttonYesGui,
                buttonNoGui, dontAskAgainName,
                options | KMessageBox::Dangerous
            );
            break;
        case KIO::JobUiDelegate::WarningYesNoCancel:
            result = KMessageBox::warningYesNoCancel(
                window(), text, caption, buttonYesGui, buttonNoGui,
                KStandardGuiItem::cancel(), dontAskAgainName, options
            );
            break;
        case KIO::JobUiDelegate::WarningContinueCancel: {
            result = KMessageBox::warningContinueCancel(
                window(), text, caption, buttonYesGui,
                KStandardGuiItem::cancel(), dontAskAgainName, options
            );
            break;
        }
        case KIO::JobUiDelegate::Information: {
            KMessageBox::information(window(), text, caption, dontAskAgainName, options);
            result = 1; // whatever
            break;
        }
        default: {
            kWarning() << "Unknown type" << type;
            result = 0;
            break;
        }
    }
    KMessageBox::setDontShowAskAgainConfig(0);
    return result;
}

void KIO::JobUiDelegate::showErrorMessage()
{
    KJob* kjob = job();
    if (kjob->error() != KJob::KilledJobError) {
        KIO::SimpleJob* kiosimplejob = qobject_cast<KIO::SimpleJob*>(kjob);
        if (kiosimplejob) {
            const KUrl kiosimplejoburl = kiosimplejob->url();
            const QStringList errors = kiosimplejob->detailedErrorStrings(&kiosimplejoburl);
            // qDebug() << Q_FUNC_INFO << errors;
            Q_ASSERT(errors.size() == 3);
            KMessageBox::detailedError(
                window(), errors[1], errors[2], errors[0],
                KMessageBox::Notify | KMessageBox::AllowLink
            );
            return;
        }

        KIO::Job* kiojob = qobject_cast<KIO::Job*>(kjob);
        if (kiojob) {
            const QStringList errors = kiojob->detailedErrorStrings();
            Q_ASSERT(errors.size() == 3);
            // qDebug() << Q_FUNC_INFO << errors;
            KMessageBox::detailedError(
                window(), errors[1], errors[2], errors[0],
                KMessageBox::Notify | KMessageBox::AllowLink
            );
            return;
        }

        KMessageBox::messageBox(window(), KMessageBox::Error, kjob->errorString());
    }
}

#include "moc_jobuidelegate.cpp"
