/*  This file is part of the KDE libraries
    Copyright (C) 2022 Ivailo Monev <xakepa10@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2, as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "kpasswdstore.h"
#include "kstandarddirs.h"
#include "klockfile.h"
#include "ksettings.h"

#include <QApplication>
#include <QDBusInterface>
#include <QDBusReply>
#include <QCryptographicHash>

#include <sys/types.h>
#include <unistd.h>

static QByteArray getCookie()
{
    KSettings ksettings("kpasswdstorerc", KSettings::SimpleConfig);
    const QString cookietype = ksettings.string("KPasswdStore/Cookie", QString()).toLower();
    if (cookietype == QLatin1String("pid")) {
        return QByteArray::number(::getpid());
    } else if (cookietype == QLatin1String("random")) {
        return qRandomUuid();
    }
    return QByteArray::number(::getuid());
}

static QString getLockName(const QByteArray &cookie, const QString &storeid)
{
    return QString::fromLatin1("%1-%2").arg(cookie, storeid);
}

class KPasswdStorePrivate
{
public:
    KPasswdStorePrivate(KPasswdStore *q);

    void ensureInterface();

    KPasswdStore *q_ptr;
    QDBusInterface *interface;
    QByteArray cookie;
    QString storeid;
};

KPasswdStorePrivate::KPasswdStorePrivate(KPasswdStore *q)
    : q_ptr(q),
    interface(nullptr),
    cookie(getCookie()),
    storeid(QApplication::applicationName())
{
}

void KPasswdStorePrivate::ensureInterface()
{
    if (!interface) {
        interface = new QDBusInterface(
            QString::fromLatin1("org.kde.kded"),
            QString::fromLatin1("/modules/kpasswdstore"),
            QString::fromLatin1("org.kde.kpasswdstore"),
            QDBusConnection::sessionBus(),
            q_ptr
        );
    }
}


KPasswdStore::KPasswdStore(QObject *parent)
    : QObject(parent),
    d(new KPasswdStorePrivate(this))
{
}

KPasswdStore::~KPasswdStore()
{
    delete d;
}

void KPasswdStore::setStoreID(const QString &id)
{
    d->storeid = id;
}

bool KPasswdStore::isOpen() const
{
    d->ensureInterface();
    KLockFile klockfile(getLockName(d->cookie, d->storeid));
    klockfile.lock();
    QDBusReply<bool> result = d->interface->call("isOpen", d->cookie, d->storeid);
    klockfile.unlock();
    return result.value();
}

bool KPasswdStore::openStore(const qlonglong windowid)
{
    d->ensureInterface();
    KLockFile klockfile(getLockName(d->cookie, d->storeid));
    klockfile.lock();
    QDBusReply<bool> result = d->interface->call("openStore", d->cookie, d->storeid, windowid);
    klockfile.unlock();
    return result.value();
}

void KPasswdStore::setCacheOnly(const bool cacheonly)
{
    d->ensureInterface();
    KLockFile klockfile(getLockName(d->cookie, d->storeid));
    klockfile.lock();
    d->interface->call("setCacheOnly", d->cookie, d->storeid, cacheonly);
    klockfile.unlock();
}

bool KPasswdStore::cacheOnly() const
{
    d->ensureInterface();
    KLockFile klockfile(getLockName(d->cookie, d->storeid));
    klockfile.lock();
    QDBusReply<bool> result = d->interface->call("cacheOnly", d->cookie, d->storeid);
    klockfile.unlock();
    return result.value();
}

QString KPasswdStore::getPasswd(const QByteArray &key, const qlonglong windowid)
{
    d->ensureInterface();
    KLockFile klockfile(getLockName(d->cookie, d->storeid));
    klockfile.lock();
    QDBusReply<QString> result = d->interface->call("getPasswd", d->cookie, d->storeid, key, windowid);
    return result.value();
}

bool KPasswdStore::storePasswd(const QByteArray &key, const QString &passwd, const qlonglong windowid)
{
    d->ensureInterface();
    KLockFile klockfile(getLockName(d->cookie, d->storeid));
    klockfile.lock();
    QDBusReply<bool> result = d->interface->call("storePasswd", d->cookie, d->storeid, key, passwd, windowid);
    klockfile.unlock();
    return result.value();
}

QStringList KPasswdStore::stores()
{
    KSettings passwdstore(KStandardDirs::locateLocal("data", "kpasswdstore"), KSettings::SimpleConfig);
    passwdstore.beginGroup("KPasswdStore");
    return passwdstore.groupKeys();
}

QByteArray KPasswdStore::makeKey(const QString &string)
{
    return QCryptographicHash::hash(string.toUtf8()).toHex();
}

#include "moc_kpasswdstore.cpp"
