 /*
 *   Copyright 2005 by Aaron Seigo <aseigo@kde.org>
 *   Copyright 2007 by Riccardo Iaconelli <riccardo@kde.org>
 *   Copyright 2008 by Ménard Alexis <darktears31@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef PLASMA_APPLET_P_H
#define PLASMA_APPLET_P_H

#include <QBasicTimer>
#include <QGraphicsProxyWidget>

#include <kactioncollection.h>
#include <kconfigdialog.h>

#include "plasma/animator.h"
#include "plasma/private/applethandle_p.h"

class KKeySequenceWidget;

namespace Plasma
{

class Dialog;
class FrameSvg;
class Wallpaper;
class BusyWidget;
class PushButton;

class AppletConfigDialog : public KConfigDialog
{
    Q_OBJECT

public:
    AppletConfigDialog(QWidget* parent, const QString &id, KConfigSkeleton *s)
        : KConfigDialog(parent, id, s),
          m_changed(false)
    {
    }

public Q_SLOTS:
    void settingsModified(bool modified = true)
    {
        m_changed = modified;
        updateButtons();
    }

protected:
    bool hasChanged()
    {
        return m_changed || KConfigDialog::hasChanged();
    }

private:
    bool m_changed;
};

class AppletOverlayWidget : public QGraphicsWidget
{
    Q_OBJECT

public:
    AppletOverlayWidget(QGraphicsWidget *parent);
    void destroy();

    qreal opacity;

protected:
    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget = 0);
    void mousePressEvent(QGraphicsSceneMouseEvent *event);

protected Q_SLOTS:
    void overlayAnimationComplete();
};

class AppletPrivate
{
public:
    AppletPrivate(KService::Ptr service, const KPluginInfo *info, int uniqueID, Applet *applet);
    ~AppletPrivate();

    void init();

    /**
     * Sets whether or not this Applet is acting as a Containment
     */
    void setIsContainment(bool isContainment, bool forceUpdate = false);

    void handleDisappeared(AppletHandle *handle);
    QString globalName() const;
    void scheduleConstraintsUpdate(Plasma::Constraints c);
    void scheduleModificationNotification();
    KConfigGroup *mainConfigGroup();
    QString visibleFailureText(const QString &reason);
    void themeChanged();
    void resetConfigurationObject();
    void selectItemToDestroy();
    void updateRect(const QRectF &rect);
    void setFocus();
    void cleanUpAndDelete();
    void createMessageOverlay(bool usePopup = true);
    void positionMessageOverlay();
    void destroyMessageOverlay();
    void addGlobalShortcutsPage(KConfigDialog *dialog);
    void configDialogFinished();
    KConfigDialog *generateGenericConfigDialog();
    void addStandardConfigurationPages(KConfigDialog *dialog);
    QString configDialogId() const;
    QString configWindowTitle() const;
    void updateShortcuts();
    void globalShortcutChanged();
    void propagateConfigChanged();
    void createActivationAction();

    static KActionCollection* defaultActions(QObject *parent);
    static QStringList knownCategories();
    static QString parentAppConstraint(const QString &parentApp = QString());

    static uint s_maxAppletId;
    static int s_maxZValue;
    static int s_minZValue;

    //TODO: examine the usage of memory here; there's a pretty large
    //      number of members at this point.
    uint appletId;
    Applet *q;

    // applet attributes
    Applet::BackgroundHints preferredBackgroundHints;
    Applet::BackgroundHints backgroundHints;
    Plasma::AspectRatioMode aspectRatioMode;
    ImmutabilityType immutability;

    // applet info we keep around in case its needed
    KPluginInfo appletDescription;
    QVariantList args;

    // bookkeeping
    QSet<QGraphicsItem*> registeredAsDragHandle;
    Plasma::FrameSvg *background;
    KConfigGroup *mainConfig;
    Plasma::Constraints pendingConstraints;

    // overlays and messages
    QWeakPointer<Plasma::Dialog> messageDialog;
    AppletOverlayWidget *messageOverlay;
    QGraphicsProxyWidget *messageOverlayProxy;
    Plasma::BusyWidget *busyWidget;
    QWeakPointer<Plasma::PushButton> messageOkButton;
    QWeakPointer<Plasma::PushButton> messageYesButton;
    QWeakPointer<Plasma::PushButton> messageNoButton;
    QWeakPointer<Plasma::PushButton> messageCancelButton;
    QWeakPointer<QAction> messageCloseAction;

    // package stuff
    QWeakPointer<AppletHandle> handle;

    // actions stuff; put activationAction into actions?
    KActionCollection *actions;
    KAction *activationAction;

    // configuration
    QWeakPointer<KKeySequenceWidget> shortcutEditor; //TODO: subclass KConfigDialog and encapsulate this in there

    ItemStatus itemStatus;

    // the applet can change size policy by itself, so save the old one for eventual restore
    QSizePolicy preferredSizePolicy;

    //keep last sizes for formfactors, useful when the containment changes
    QHash<FormFactor, QSizeF> sizeForFormFactor;

    // timerEvent bookkeeping
    QBasicTimer constraintsTimer;
    QBasicTimer busyWidgetTimer;
    QBasicTimer *modificationsTimer;

    // a great green field of booleans :)
    bool hasConfigurationInterface;
    bool failed;
    bool isContainment;
    bool transient;
    bool needsConfig;
    bool started;
};

} // Plasma namespace

#endif
