/*
    Copyright 2010 Rafael Fernández López <ereslibre@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) version 3, or any
    later version accepted by the membership of KDE e.V. (or its
    successor approved by the membership of KDE e.V.), which shall
    act as a proxy defined in Section 6 of version 3 of the license.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library. If not, see <http://www.gnu.org/licenses/>.
*/

#include "udevprocessor.h"
#include "udevdevice.h"
#include "cpuinfo.h"

#include <QFile>
#include <QDir>
#include <qmath.h>

using namespace Solid::Backends::UDev;

Processor::Processor(UDevDevice *device)
    : DeviceInterface(device),
    m_minSpeed(-1),
    m_maxSpeed(-1)
{
}

Processor::~Processor()
{
}

int Processor::number() const
{
    // There's a subtle assumption here: suppose the system's ACPI
    // supports more processors/cores than are installed, and so udev reports
    // 4 cores when there are 2, say.  Will the processor numbers (in
    // /proc/cpuinfo, in particular) always match the sysfs device numbers?
    return m_device->deviceNumber();
}

// NOTE: do not parse /proc/cpuinfo for "cpu MHz", that may be current not maximum speed
int Processor::minSpeed() const
{
    if (m_minSpeed == -1) {
        m_minSpeed = getCPUInfo("/cpufreq/cpuinfo_min_freq");
    }
    return m_minSpeed;
}

int Processor::maxSpeed() const
{
    if (m_maxSpeed == -1) {
        m_maxSpeed = getCPUInfo("/cpufreq/cpuinfo_max_freq");
    }
    return m_maxSpeed;
}

bool Processor::canChangeFrequency() const
{
    return (minSpeed() > 0 && maxSpeed() > 0);
}

Solid::Processor::InstructionSets Processor::instructionSets() const
{
    static QStringList cpuflags = extractCpuInfoLine(m_device->deviceNumber(), "flags\\s+:\\s+(\\S.+)").split(" ");

    // for reference:
    // arch/x86/include/asm/cpufeatures.h
    // arch/powerpc/kernel/prom.c
    Solid::Processor::InstructionSets cpuinstructions = Solid::Processor::NoExtensions;
    if (cpuflags.contains("mmx")) {
        cpuinstructions |= Solid::Processor::IntelMmx;
    }
    if (cpuflags.contains("sse")) {
        cpuinstructions |= Solid::Processor::IntelSse;
    }
    if (cpuflags.contains("sse2")) {
        cpuinstructions |= Solid::Processor::IntelSse2;
    }
    if (cpuflags.contains("pni") || cpuflags.contains("ssse3")) {
        cpuinstructions |= Solid::Processor::IntelSse3;
    }
    if (cpuflags.contains("sse4") || cpuflags.contains("sse4_1") || cpuflags.contains("sse4_2")) {
        cpuinstructions |= Solid::Processor::IntelSse4;
    }
    if (cpuflags.contains("3dnow") || cpuflags.contains("3dnowext")) {
        cpuinstructions |= Solid::Processor::Amd3DNow;
    }
    if (cpuflags.contains("altivec")) {
        cpuinstructions |= Solid::Processor::AltiVec;
    }

    return cpuinstructions;
}

int Processor::getCPUInfo(const char* filename) const
{
    static const QLatin1String sysPrefix("/sysdev");

    QString cpuFreqFileName(m_device->deviceName());
    if (QDir(cpuFreqFileName + sysPrefix).exists()) {
        cpuFreqFileName.append(sysPrefix);
    }
    cpuFreqFileName.append(QLatin1String(filename));

    QFile cpuFreqFile(cpuFreqFileName);
    if (cpuFreqFile.open(QIODevice::ReadOnly)) {
        const qlonglong value = cpuFreqFile.readAll().trimmed().toLongLong();
        if (value > 0) {
            // value is in kHz
            return qRound(value / 1000);
        }
    }
    return 0;
}

#include "backends/udev/moc_udevprocessor.cpp"
