/****************************************************************************
**
** Copyright (C) 2015 The Qt Company Ltd.
** Copyright (C) 2016 Ivailo Monev
**
** This file is part of the QtCore module of the Katie Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
**
** GNU Lesser General Public License Usage
** This file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef QLOCALE_H
#define QLOCALE_H

#include <QtCore/qvariant.h>


QT_BEGIN_NAMESPACE

class QDataStream;
class QDate;
class QDateTime;
class QTime;

class QLocalePrivate;

class Q_CORE_EXPORT QLocale
{
    Q_GADGET
    Q_ENUMS(Language)
    Q_ENUMS(Country)
    friend class QString;
    friend class QByteArray;
    friend class QIntValidator;
    friend class QDoubleValidatorPrivate;
    friend class QTextStream;
    friend class QTextStreamPrivate;

public:
    // GENERATED PART STARTS HERE
    // see genlocale.py for more info on generated data
    enum Language {
        AnyLanguage = 0,
        C = 1,
        Abkhazian = 2,
        Achinese = 3,
        Acoli = 4,
        Adangme = 5,
        Adyghe = 6,
        Afar = 7,
        Afrihili = 8,
        Afrikaans = 9,
        Aghem = 10,
        Ainu = 11,
        Akan = 12,
        Akkadian = 13,
        Akoose = 14,
        Alabama = 15,
        Albanian = 16,
        Aleut = 17,
        AlgerianArabic = 18,
        AmericanEnglish = 19,
        AmericanSignLanguage = 20,
        Amharic = 21,
        AncientEgyptian = 22,
        AncientGreek = 23,
        Angika = 24,
        Anii = 25,
        AoNaga = 26,
        Arabic = 27,
        ArabicNajdi = 28,
        Aragonese = 29,
        Aramaic = 30,
        Araona = 31,
        Arapaho = 32,
        Arawak = 33,
        Armenian = 34,
        Aromanian = 35,
        Arpitan = 36,
        Assamese = 37,
        Asturian = 38,
        Asu = 39,
        Atikamekw = 40,
        Atsam = 41,
        AustralianEnglish = 42,
        AustrianGerman = 43,
        Avaric = 44,
        Avestan = 45,
        Awadhi = 46,
        Aymara = 47,
        Azerbaijani = 48,
        Badaga = 49,
        Bafia = 50,
        Bafut = 51,
        Bakhtiari = 52,
        Balinese = 53,
        Baluchi = 54,
        Bambara = 55,
        Bamun = 56,
        Bangla = 57,
        Banjar = 58,
        Basaa = 59,
        Bashkir = 60,
        Basque = 61,
        BatakToba = 62,
        Bavarian = 63,
        Beja = 64,
        Belarusian = 65,
        Bemba = 66,
        Bena = 67,
        Betawi = 68,
        Bhojpuri = 69,
        Bikol = 70,
        Bini = 71,
        Bishnupriya = 72,
        Bislama = 73,
        Blin = 74,
        Blissymbols = 75,
        Bodo = 76,
        Bosnian = 77,
        Brahui = 78,
        Braj = 79,
        BrazilianPortuguese = 80,
        Breton = 81,
        BritishEnglish = 82,
        Buginese = 83,
        Bulgarian = 84,
        Bulu = 85,
        Buriat = 86,
        Burmese = 87,
        Caddo = 88,
        CajunFrench = 89,
        CanadianEnglish = 90,
        CanadianFrench = 91,
        Cantonese = 92,
        Capiznon = 93,
        Carib = 94,
        CarolinaAlgonquian = 95,
        Catalan = 96,
        Cayuga = 97,
        Cebuano = 98,
        CentralAtlasTamazight = 99,
        CentralDusun = 100,
        CentralKurdish = 101,
        CentralOjibwa = 102,
        CentralYupik = 103,
        ChadianArabic = 104,
        Chagatai = 105,
        Chakma = 106,
        Chamorro = 107,
        Chechen = 108,
        Cherokee = 109,
        Cheyenne = 110,
        Chibcha = 111,
        Chickasaw = 112,
        Chiga = 113,
        Chilcotin = 114,
        ChimborazoHighlandQuichua = 115,
        Chinese = 116,
        ChinookJargon = 117,
        Chipewyan = 118,
        Choctaw = 119,
        ChurchSlavic = 120,
        Chuukese = 121,
        Chuvash = 122,
        ClassicalNewari = 123,
        ClassicalSyriac = 124,
        Colognian = 125,
        Comorian = 126,
        CongoSwahili = 127,
        Coptic = 128,
        Cornish = 129,
        Corsican = 130,
        Cree = 131,
        CrimeanTatar = 132,
        Croatian = 133,
        Czech = 134,
        Dakota = 135,
        Danish = 136,
        Dargwa = 137,
        Dari = 138,
        Dazaga = 139,
        Delaware = 140,
        Dinka = 141,
        Divehi = 142,
        Dogri = 143,
        Dogrib = 144,
        Duala = 145,
        Dutch = 146,
        Dyula = 147,
        Dzongkha = 148,
        EasternCanadianInuktitut = 149,
        EasternFrisian = 150,
        EasternOjibwa = 151,
        Efik = 152,
        EgyptianArabic = 153,
        Ekajuk = 154,
        Elamite = 155,
        Embu = 156,
        Emilian = 157,
        English = 158,
        Erzya = 159,
        Esperanto = 160,
        Estonian = 161,
        EuropeanPortuguese = 162,
        EuropeanSpanish = 163,
        Ewe = 164,
        Ewondo = 165,
        Extremaduran = 166,
        Fang = 167,
        Fanti = 168,
        Faroese = 169,
        FijiHindi = 170,
        Fijian = 171,
        Filipino = 172,
        Finnish = 173,
        Flemish = 174,
        Fon = 175,
        Frafra = 176,
        French = 177,
        Friulian = 178,
        Fula = 179,
        Ga = 180,
        Gagauz = 181,
        Galician = 182,
        GanChinese = 183,
        Ganda = 184,
        Gayo = 185,
        Gbaya = 186,
        Geez = 187,
        Georgian = 188,
        German = 189,
        GhegAlbanian = 190,
        Ghomala = 191,
        Gilaki = 192,
        Gilbertese = 193,
        GoanKonkani = 194,
        Gondi = 195,
        Gorontalo = 196,
        Gothic = 197,
        Grebo = 198,
        Greek = 199,
        Guarani = 200,
        Gujarati = 201,
        Gusii = 202,
        Gwichin = 203,
        Haida = 204,
        HaitianCreole = 205,
        HakkaChinese = 206,
        Halkomelem = 207,
        Haryanvi = 208,
        Hausa = 209,
        Hawaiian = 210,
        Hebrew = 211,
        Herero = 212,
        Hiligaynon = 213,
        Hindi = 214,
        HindiLatin = 215,
        HiriMotu = 216,
        Hittite = 217,
        Hmong = 218,
        HmongNjua = 219,
        Hungarian = 220,
        Hupa = 221,
        Iban = 222,
        Ibibio = 223,
        Icelandic = 224,
        Ido = 225,
        Igbo = 226,
        Iloko = 227,
        InariSami = 228,
        Indonesian = 229,
        Ingrian = 230,
        Ingush = 231,
        Innuaimun = 232,
        Interlingua = 233,
        Interlingue = 234,
        Inuktitut = 235,
        Inupiaq = 236,
        Irish = 237,
        Italian = 238,
        JamaicanCreoleEnglish = 239,
        Japanese = 240,
        Javanese = 241,
        Jju = 242,
        JolaFonyi = 243,
        JudeoArabic = 244,
        JudeoPersian = 245,
        Jutish = 246,
        Kabardian = 247,
        Kabuverdianu = 248,
        Kabyle = 249,
        Kachin = 250,
        Kaingang = 251,
        Kako = 252,
        Kalaallisut = 253,
        Kalenjin = 254,
        Kalmyk = 255,
        Kamba = 256,
        Kanembu = 257,
        Kangri = 258,
        Kannada = 259,
        Kanuri = 260,
        KaraKalpak = 261,
        KarachayBalkar = 262,
        Karelian = 263,
        Kashmiri = 264,
        Kashubian = 265,
        Kawi = 266,
        Kazakh = 267,
        Kenyang = 268,
        Khasi = 269,
        Khmer = 270,
        Khotanese = 271,
        Khowar = 272,
        Kiche = 273,
        Kikuyu = 274,
        Kimbundu = 275,
        Kinaraya = 276,
        Kinyarwanda = 277,
        Kirghiz = 278,
        Kirmanjki = 279,
        Klingon = 280,
        Kom = 281,
        Komi = 282,
        KomiPermyak = 283,
        Kongo = 284,
        Konkani = 285,
        Korean = 286,
        Koro = 287,
        Kosraean = 288,
        Kotava = 289,
        KoyraChiini = 290,
        KoyraboroSenni = 291,
        Kpelle = 292,
        Krio = 293,
        Kuanyama = 294,
        Kumyk = 295,
        Kurdish = 296,
        Kurukh = 297,
        Kutenai = 298,
        Kuvi = 299,
        Kwakwala = 300,
        Kwasio = 301,
        Ladino = 302,
        Lakota = 303,
        Lamba = 304,
        Langi = 305,
        Lao = 306,
        Latgalian = 307,
        Latin = 308,
        LatinAmericanSpanish = 309,
        Latvian = 310,
        Laz = 311,
        Lezghian = 312,
        Ligurian = 313,
        Lillooet = 314,
        Limburgish = 315,
        Lingala = 316,
        LinguaFrancaNova = 317,
        LiteraryChinese = 318,
        Lithuanian = 319,
        Livonian = 320,
        Lojban = 321,
        Lombard = 322,
        LouisianaCreole = 323,
        LowGerman = 324,
        LowSaxon = 325,
        LowerSilesian = 326,
        LowerSorbian = 327,
        Lozi = 328,
        LubaKatanga = 329,
        LubaLulua = 330,
        Luiseno = 331,
        LuleSami = 332,
        Lunda = 333,
        Luo = 334,
        Luxembourgish = 335,
        Luyia = 336,
        Maba = 337,
        Macedonian = 338,
        Machame = 339,
        Madurese = 340,
        Mafa = 341,
        Magahi = 342,
        MainFranconian = 343,
        Maithili = 344,
        Makasar = 345,
        Makhuwa = 346,
        MakhuwaMeetto = 347,
        Makonde = 348,
        Malagasy = 349,
        Malay = 350,
        Malayalam = 351,
        MaliseetPassamaquoddy = 352,
        Maltese = 353,
        Manchu = 354,
        Mandar = 355,
        Mandingo = 356,
        Manipuri = 357,
        Manx = 358,
        Maori = 359,
        Mapuche = 360,
        Marathi = 361,
        Mari = 362,
        Marshallese = 363,
        Marwari = 364,
        Masai = 365,
        Mazanderani = 366,
        Medumba = 367,
        Mende = 368,
        Mentawai = 369,
        Meru = 370,
        Meta = 371,
        MexicanSpanish = 372,
        Michif = 373,
        MiddleDutch = 374,
        MiddleEnglish = 375,
        MiddleFrench = 376,
        MiddleHighGerman = 377,
        MiddleIrish = 378,
        Mikmaw = 379,
        MinNanChinese = 380,
        Minangkabau = 381,
        Mingrelian = 382,
        Mirandese = 383,
        Mizo = 384,
        ModernStandardArabic = 385,
        Mohawk = 386,
        Moksha = 387,
        Moldavian = 388,
        Mongo = 389,
        Mongolian = 390,
        Montenegrin = 391,
        MooseCree = 392,
        Morisyen = 393,
        MoroccanArabic = 394,
        Mossi = 395,
        Multiplelanguages = 396,
        Mundang = 397,
        Muscogee = 398,
        MuslimTat = 399,
        Myene = 400,
        NKo = 401,
        Nama = 402,
        NauruLanguage = 403,
        Navajo = 404,
        Ndonga = 405,
        Neapolitan = 406,
        Nepali = 407,
        Newari = 408,
        Ngambay = 409,
        Ngiemboon = 410,
        Ngomba = 411,
        Nheengatu = 412,
        Nias = 413,
        NigerianPidgin = 414,
        Niuean = 415,
        Nogai = 416,
        Nolinguisticcontent = 417,
        NorthNdebele = 418,
        NorthernEastCree = 419,
        NorthernFrisian = 420,
        NorthernHaida = 421,
        NorthernLuri = 422,
        NorthernSami = 423,
        NorthernSotho = 424,
        NorthernTutchone = 425,
        NorthwesternOjibwa = 426,
        Norwegian = 427,
        NorwegianBokmal = 428,
        NorwegianNynorsk = 429,
        Novial = 430,
        Nuer = 431,
        Nyamwezi = 432,
        Nyanja = 433,
        Nyankole = 434,
        NyasaTonga = 435,
        Nyoro = 436,
        Nzima = 437,
        Obolo = 438,
        Occitan = 439,
        Odia = 440,
        OjiCree = 441,
        Ojibwa = 442,
        Okanagan = 443,
        OldEnglish = 444,
        OldFrench = 445,
        OldHighGerman = 446,
        OldIrish = 447,
        OldNorse = 448,
        OldPersian = 449,
        OldProvencal = 450,
        Oromo = 451,
        Osage = 452,
        Ossetic = 453,
        OttomanTurkish = 454,
        Pahlavi = 455,
        PalatineGerman = 456,
        Palauan = 457,
        Pali = 458,
        Pampanga = 459,
        Pangasinan = 460,
        Papiamento = 461,
        Pashto = 462,
        PennsylvaniaGerman = 463,
        Persian = 464,
        Phoenician = 465,
        Picard = 466,
        Piedmontese = 467,
        Pijin = 468,
        PlainsCree = 469,
        Plautdietsch = 470,
        Pohnpeian = 471,
        Polish = 472,
        Pontic = 473,
        Portuguese = 474,
        Prussian = 475,
        Punjabi = 476,
        Quechua = 477,
        Rajasthani = 478,
        Rapanui = 479,
        Rarotongan = 480,
        Riffian = 481,
        Rohingya = 482,
        Romagnol = 483,
        Romanian = 484,
        Romansh = 485,
        Romany = 486,
        Rombo = 487,
        Rotuman = 488,
        Roviana = 489,
        Rundi = 490,
        Russian = 491,
        Rusyn = 492,
        Rwa = 493,
        Saamia = 494,
        Saho = 495,
        SamaritanAramaic = 496,
        Samburu = 497,
        SamiSkolt = 498,
        SamiSouthern = 499,
        Samoan = 500,
        Samogitian = 501,
        Sandawe = 502,
        Sango = 503,
        Sangu = 504,
        Sanskrit = 505,
        Santali = 506,
        Sardinian = 507,
        Sasak = 508,
        SassareseSardinian = 509,
        SaterlandFrisian = 510,
        Saurashtra = 511,
        Scots = 512,
        ScottishGaelic = 513,
        Selayar = 514,
        Selkup = 515,
        Sena = 516,
        Seneca = 517,
        Serbian = 518,
        SerboCroatian = 519,
        Serer = 520,
        Seri = 521,
        SeselwaCreoleFrench = 522,
        Shambala = 523,
        Shan = 524,
        Shona = 525,
        SichuanYi = 526,
        Sicilian = 527,
        Sidamo = 528,
        Siksika = 529,
        Silesian = 530,
        SimplifiedChinese = 531,
        Sindhi = 532,
        Sinhala = 533,
        Slave = 534,
        Slovak = 535,
        Slovenian = 536,
        Soga = 537,
        Sogdien = 538,
        Somali = 539,
        Soninke = 540,
        SouthNdebele = 541,
        SouthernAltai = 542,
        SouthernEastCree = 543,
        SouthernHaida = 544,
        SouthernKurdish = 545,
        SouthernLushootseed = 546,
        SouthernSotho = 547,
        SouthernTutchone = 548,
        Spanish = 549,
        SrananTongo = 550,
        StandardMoroccanTamazight = 551,
        StraitsSalish = 552,
        Sukuma = 553,
        Sumerian = 554,
        Sundanese = 555,
        Susu = 556,
        Swahili = 557,
        SwampyCree = 558,
        Swati = 559,
        Swedish = 560,
        SwissFrench = 561,
        SwissGerman = 562,
        SwissHighGerman = 563,
        Syriac = 564,
        Tachelhit = 565,
        Tagalog = 566,
        Tagish = 567,
        Tahitian = 568,
        Tahltan = 569,
        TaiDam = 570,
        Taita = 571,
        Tajik = 572,
        Talysh = 573,
        Tamashek = 574,
        Tamil = 575,
        Taroko = 576,
        Tasawaq = 577,
        Tatar = 578,
        Telugu = 579,
        Tereno = 580,
        Teso = 581,
        Tetum = 582,
        Thai = 583,
        Tibetan = 584,
        Tigre = 585,
        Tigrinya = 586,
        Timne = 587,
        Tiv = 588,
        Tlingit = 589,
        TokPisin = 590,
        TokelauLanguage = 591,
        TokiPona = 592,
        Tongan = 593,
        TornedalenFinnish = 594,
        Torwali = 595,
        TraditionalChinese = 596,
        Tsakhur = 597,
        Tsakonian = 598,
        Tsimshian = 599,
        Tsonga = 600,
        Tswana = 601,
        Tulu = 602,
        Tumbuka = 603,
        TunisianArabic = 604,
        Turkish = 605,
        Turkmen = 606,
        Turoyo = 607,
        TuvaluLanguage = 608,
        Tuvinian = 609,
        Twi = 610,
        Tyap = 611,
        Udmurt = 612,
        Ugaritic = 613,
        Uighur = 614,
        Ukrainian = 615,
        Umbundu = 616,
        Unknownlanguage = 617,
        UpperSorbian = 618,
        Urdu = 619,
        Uzbek = 620,
        Vai = 621,
        Venda = 622,
        Venetian = 623,
        Veps = 624,
        Vietnamese = 625,
        Volapuk = 626,
        Voro = 627,
        Votic = 628,
        Vunjo = 629,
        Walloon = 630,
        Walser = 631,
        Waray = 632,
        Warlpiri = 633,
        Washo = 634,
        Wayuu = 635,
        Welsh = 636,
        WestFlemish = 637,
        WesternBalochi = 638,
        WesternCanadianInuktitut = 639,
        WesternFrisian = 640,
        WesternMari = 641,
        WesternOjibwa = 642,
        WesternPanjabi = 643,
        Wolaytta = 644,
        Wolof = 645,
        WoodsCree = 646,
        WuChinese = 647,
        Xhosa = 648,
        XiangChinese = 649,
        Yakut = 650,
        Yangben = 651,
        Yao = 652,
        Yapese = 653,
        Yemba = 654,
        Yiddish = 655,
        Yoruba = 656,
        Zapotec = 657,
        Zarma = 658,
        Zaza = 659,
        Zeelandic = 660,
        Zenaga = 661,
        Zhuang = 662,
        ZoroastrianDari = 663,
        Zulu = 664,
        Zuni = 665,

        Azeri = Azerbaijani,
        ChineseCantonese = Cantonese,
        ChineseMandarin = Chinese,
        Hinglish = HindiLatin,
        KurdishCentral = CentralKurdish,
        KurdishSorani = CentralKurdish,
        Kyrgyz = Kirghiz,
        MandarinChinese = Chinese,
        MyanmarLanguage = Burmese,
        NajdiArabic = ArabicNajdi,
        Pushto = Pashto,
        SamiInari = InariSami,
        SamiLule = LuleSami,
        SamiNorthern = NorthernSami,
        SimplifiedMandarinChinese = SimplifiedChinese,
        SkoltSami = SamiSkolt,
        SouthernSami = SamiSouthern,
        TraditionalMandarinChinese = TraditionalChinese,
        UKEnglish = BritishEnglish,
        USEnglish = AmericanEnglish,
        Uyghur = Uighur,

        LastLanguage = Zuni
    };

    enum Country {
        AnyCountry = 0,
        Afghanistan = 1,
        Africa = 2,
        AlandIslands = 3,
        Albania = 4,
        Algeria = 5,
        AmericanSamoa = 6,
        Americas = 7,
        Andorra = 8,
        Angola = 9,
        Anguilla = 10,
        Antarctica = 11,
        AntiguaAndBarbuda = 12,
        AotearoaNewZealand = 13,
        Argentina = 14,
        Armenia = 15,
        Aruba = 16,
        AscensionIsland = 17,
        Asia = 18,
        Australasia = 19,
        Australia = 20,
        Austria = 21,
        Azerbaijan = 22,
        Bahamas = 23,
        Bahrain = 24,
        Bangladesh = 25,
        Barbados = 26,
        Belarus = 27,
        Belgium = 28,
        Belize = 29,
        Benin = 30,
        Bermuda = 31,
        Bhutan = 32,
        Bolivia = 33,
        Bosnia = 34,
        Botswana = 35,
        BouvetIsland = 36,
        Brazil = 37,
        BritishIndianOceanTerritory = 38,
        BritishVirginIslands = 39,
        Brunei = 40,
        Bulgaria = 41,
        BurkinaFaso = 42,
        Burundi = 43,
        CaboVerde = 44,
        Cambodia = 45,
        Cameroon = 46,
        Canada = 47,
        CanaryIslands = 48,
        Caribbean = 49,
        CaribbeanNetherlands = 50,
        CaymanIslands = 51,
        CentralAfricanRepublic = 52,
        CentralAmerica = 53,
        CentralAsia = 54,
        CeutaAndMelilla = 55,
        Chad = 56,
        Chile = 57,
        China = 58,
        ChristmasIsland = 59,
        ClippertonIsland = 60,
        CocosKeelingIslands = 61,
        Colombia = 62,
        Comoros = 63,
        CongoBrazzaville = 64,
        CongoDRC = 65,
        CookIslands = 66,
        CostaRica = 67,
        CotedIvoire = 68,
        Croatia = 69,
        Cuba = 70,
        Curacao = 71,
        Cyprus = 72,
        CzechRepublic = 73,
        Denmark = 74,
        DiegoGarcia = 75,
        Djibouti = 76,
        Dominica = 77,
        DominicanRepublic = 78,
        EastTimor = 79,
        EasternAfrica = 80,
        EasternAsia = 81,
        EasternEurope = 82,
        Ecuador = 83,
        Egypt = 84,
        ElSalvador = 85,
        EquatorialGuinea = 86,
        Eritrea = 87,
        Estonia = 88,
        Eswatini = 89,
        Ethiopia = 90,
        Europe = 91,
        EuropeanUnion = 92,
        Eurozone = 93,
        FalklandIslands = 94,
        FaroeIslands = 95,
        Fiji = 96,
        Finland = 97,
        France = 98,
        FrenchGuiana = 99,
        FrenchPolynesia = 100,
        FrenchSouthernTerritories = 101,
        Gabon = 102,
        Gambia = 103,
        Georgia = 104,
        Germany = 105,
        Ghana = 106,
        Gibraltar = 107,
        Greece = 108,
        Greenland = 109,
        Grenada = 110,
        Guadeloupe = 111,
        Guam = 112,
        Guatemala = 113,
        Guernsey = 114,
        Guinea = 115,
        GuineaBissau = 116,
        Guyana = 117,
        Haiti = 118,
        HeardAndMcDonaldIslands = 119,
        Honduras = 120,
        HongKong = 121,
        Hungary = 122,
        Iceland = 123,
        India = 124,
        Indonesia = 125,
        Iran = 126,
        Iraq = 127,
        Ireland = 128,
        IsleofMan = 129,
        Israel = 130,
        Italy = 131,
        Jamaica = 132,
        Japan = 133,
        Jersey = 134,
        Jordan = 135,
        Kazakhstan = 136,
        Kenya = 137,
        Kiribati = 138,
        Kosovo = 139,
        Kuwait = 140,
        Kyrgyzstan = 141,
        Laos = 142,
        LatinAmerica = 143,
        Latvia = 144,
        Lebanon = 145,
        Lesotho = 146,
        Liberia = 147,
        Libya = 148,
        Liechtenstein = 149,
        Lithuania = 150,
        Luxembourg = 151,
        Macao = 152,
        Madagascar = 153,
        Malawi = 154,
        Malaysia = 155,
        Maldives = 156,
        Mali = 157,
        Malta = 158,
        MarshallIslands = 159,
        Martinique = 160,
        Mauritania = 161,
        Mauritius = 162,
        Mayotte = 163,
        Melanesia = 164,
        Mexico = 165,
        Micronesia = 166,
        MicronesianRegion = 167,
        MiddleAfrica = 168,
        Moldova = 169,
        Monaco = 170,
        Mongolia = 171,
        Montenegro = 172,
        Montserrat = 173,
        Morocco = 174,
        Mozambique = 175,
        Myanmar = 176,
        Namibia = 177,
        Nauru = 178,
        Nepal = 179,
        Netherlands = 180,
        NewCaledonia = 181,
        Nicaragua = 182,
        Niger = 183,
        Nigeria = 184,
        Niue = 185,
        NorfolkIsland = 186,
        NorthAmerica = 187,
        NorthKorea = 188,
        NorthMacedonia = 189,
        NorthernAfrica = 190,
        NorthernAmerica = 191,
        NorthernEurope = 192,
        NorthernMarianaIslands = 193,
        Norway = 194,
        Oceania = 195,
        Oman = 196,
        OutlyingOceania = 197,
        Pakistan = 198,
        Palau = 199,
        Palestine = 200,
        Panama = 201,
        PapuaNewGuinea = 202,
        Paraguay = 203,
        Peru = 204,
        Philippines = 205,
        PitcairnIslands = 206,
        Poland = 207,
        Polynesia = 208,
        Portugal = 209,
        PseudoAccents = 210,
        PseudoBidi = 211,
        PuertoRico = 212,
        Qatar = 213,
        Reunion = 214,
        Romania = 215,
        Russia = 216,
        Rwanda = 217,
        Samoa = 218,
        SanMarino = 219,
        SaoTomeAndPrincipe = 220,
        Sark = 221,
        SaudiArabia = 222,
        Senegal = 223,
        Serbia = 224,
        Seychelles = 225,
        SierraLeone = 226,
        Singapore = 227,
        SintMaarten = 228,
        Slovakia = 229,
        Slovenia = 230,
        SolomonIslands = 231,
        Somalia = 232,
        SouthAfrica = 233,
        SouthAmerica = 234,
        SouthGeorgiaAndSouthSandwichIslands = 235,
        SouthKorea = 236,
        SouthSudan = 237,
        SoutheastAsia = 238,
        SouthernAfrica = 239,
        SouthernAsia = 240,
        SouthernEurope = 241,
        Spain = 242,
        SriLanka = 243,
        StBarthelemy = 244,
        StHelena = 245,
        StKittsAndNevis = 246,
        StLucia = 247,
        StMartin = 248,
        StPierreAndMiquelon = 249,
        StVincentAndGrenadines = 250,
        SubSaharanAfrica = 251,
        Sudan = 252,
        Suriname = 253,
        SvalbardAndJanMayen = 254,
        Sweden = 255,
        Switzerland = 256,
        Syria = 257,
        Taiwan = 258,
        Tajikistan = 259,
        Tanzania = 260,
        Thailand = 261,
        Togo = 262,
        Tokelau = 263,
        Tonga = 264,
        TrinidadAndTobago = 265,
        TristandaCunha = 266,
        Tunisia = 267,
        Turkey = 268,
        Turkmenistan = 269,
        TurksAndCaicosIslands = 270,
        Tuvalu = 271,
        UK = 272,
        UN = 273,
        US = 274,
        Uganda = 275,
        Ukraine = 276,
        UnitedArabEmirates = 277,
        UnitedStatesOutlyingIslands = 278,
        UnitedStatesVirginIslands = 279,
        UnknownRegion = 280,
        Uruguay = 281,
        Uzbekistan = 282,
        Vanuatu = 283,
        VaticanCity = 284,
        Venezuela = 285,
        Vietnam = 286,
        WallisAndFutuna = 287,
        WesternAfrica = 288,
        WesternAsia = 289,
        WesternEurope = 290,
        WesternSahara = 291,
        Yemen = 292,
        Zambia = 293,
        Zimbabwe = 294,
        world = 295,

        BosniaAndHerzegovina = Bosnia,
        CapeVerde = CaboVerde,
        ChagosArchipelago = BritishIndianOceanTerritory,
        CongoKinshasa = CongoDRC,
        CongoRepublic = CongoBrazzaville,
        Czechia = CzechRepublic,
        FalklandIslandsIslasMalvinas = FalklandIslands,
        HongKongSARChina = HongKong,
        IvoryCoast = CotedIvoire,
        MacaoSARChina = Macao,
        MyanmarBurma = Myanmar,
        NewZealand = AotearoaNewZealand,
        PalestinianTerritories = Palestine,
        Swaziland = Eswatini,
        TimorLeste = EastTimor,
        Turkiye = Turkey,
        UnitedKingdom = UK,
        UnitedNations = UN,
        UnitedStates = US,

        LastCountry = world
    };

    enum Script {
        AnyScript = 0,
        AdlamScript = 1,
        AfakaScript = 2,
        AhomScript = 3,
        AnatolianHieroglyphsScript = 4,
        ArabicScript = 5,
        ArmenianScript = 6,
        AvestanScript = 7,
        BalineseScript = 8,
        BamumScript = 9,
        BanglaScript = 10,
        BassaVahScript = 11,
        BatakScript = 12,
        BhaiksukiScript = 13,
        BlissymbolsScript = 14,
        BookPahlaviScript = 15,
        BopomofoScript = 16,
        BrahmiScript = 17,
        BrailleScript = 18,
        BugineseScript = 19,
        BuhidScript = 20,
        CarianScript = 21,
        CaucasianAlbanianScript = 22,
        ChakmaScript = 23,
        ChamScript = 24,
        CherokeeScript = 25,
        ChorasmianScript = 26,
        CirthScript = 27,
        CopticScript = 28,
        CypriotScript = 29,
        CyproMinoanScript = 30,
        CyrillicScript = 31,
        DeseretScript = 32,
        DevanagariScript = 33,
        DivesAkuruScript = 34,
        DograScript = 35,
        DuployanshorthandScript = 36,
        EasternSyriacScript = 37,
        EgyptiandemoticScript = 38,
        EgyptianhieraticScript = 39,
        EgyptianhieroglyphsScript = 40,
        ElbasanScript = 41,
        ElymaicScript = 42,
        EmojiScript = 43,
        EstrangeloSyriacScript = 44,
        EthiopicScript = 45,
        FrakturLatinScript = 46,
        FraserScript = 47,
        GaelicLatinScript = 48,
        GeorgianKhutsuriScript = 49,
        GeorgianScript = 50,
        GlagoliticScript = 51,
        GothicScript = 52,
        GranthaScript = 53,
        GreekScript = 54,
        GujaratiScript = 55,
        GunjalaGondiScript = 56,
        GurmukhiScript = 57,
        HanScript = 58,
        HangulScript = 59,
        HanifiRohingyaScript = 60,
        HanunooScript = 61,
        HanwithBopomofoScript = 62,
        HatranScript = 63,
        HebrewScript = 64,
        HiraganaScript = 65,
        ImperialAramaicScript = 66,
        IndusScript = 67,
        InheritedScript = 68,
        InscriptionalPahlaviScript = 69,
        InscriptionalParthianScript = 70,
        JamoScript = 71,
        JapaneseScript = 72,
        JapanesesyllabariesScript = 73,
        JavaneseScript = 74,
        JurchenScript = 75,
        KaithiScript = 76,
        KannadaScript = 77,
        KatakanaScript = 78,
        KawiScript = 79,
        KayahLiScript = 80,
        KharoshthiScript = 81,
        KhitansmallscriptScript = 82,
        KhmerScript = 83,
        KhojkiScript = 84,
        KhudawadiScript = 85,
        KoreanScript = 86,
        KpelleScript = 87,
        LannaScript = 88,
        LaoScript = 89,
        LatinScript = 90,
        LepchaScript = 91,
        LimbuScript = 92,
        LinearAScript = 93,
        LinearBScript = 94,
        LomaScript = 95,
        LycianScript = 96,
        LydianScript = 97,
        MahajaniScript = 98,
        MakasarScript = 99,
        MalayalamScript = 100,
        MandaeanScript = 101,
        ManichaeanScript = 102,
        MarchenScript = 103,
        MasaramGondiScript = 104,
        MathematicalNotationScript = 105,
        MayanhieroglyphsScript = 106,
        MedefaidrinScript = 107,
        MeiteiMayekScript = 108,
        MendeScript = 109,
        MeroiticCursiveScript = 110,
        MeroiticScript = 111,
        ModiScript = 112,
        MongolianScript = 113,
        MoonScript = 114,
        MroScript = 115,
        MultaniScript = 116,
        MyanmarScript = 117,
        NKoScript = 118,
        NabataeanScript = 119,
        NagMundariScript = 120,
        NandinagariScript = 121,
        NastaliqScript = 122,
        NaxiGebaScript = 123,
        NewTaiLueScript = 124,
        NewaScript = 125,
        NushuScript = 126,
        NyiakengPuachueHmongScript = 127,
        OdiaScript = 128,
        OghamScript = 129,
        OlChikiScript = 130,
        OldChurchSlavonicCyrillicScript = 131,
        OldHungarianScript = 132,
        OldItalicScript = 133,
        OldNorthArabianScript = 134,
        OldPermicScript = 135,
        OldPersianScript = 136,
        OldSogdianScript = 137,
        OldSouthArabianScript = 138,
        OldUyghurScript = 139,
        OrkhonScript = 140,
        OsageScript = 141,
        OsmanyaScript = 142,
        PahawhHmongScript = 143,
        PalmyreneScript = 144,
        PauCinHauScript = 145,
        PhagspaScript = 146,
        PhoenicianScript = 147,
        PollardPhoneticScript = 148,
        PsalterPahlaviScript = 149,
        RejangScript = 150,
        RongorongoScript = 151,
        RunicScript = 152,
        SACuneiformScript = 153,
        SamaritanScript = 154,
        SaratiScript = 155,
        SaurashtraScript = 156,
        SharadaScript = 157,
        ShavianScript = 158,
        SiddhamScript = 159,
        SignWritingScript = 160,
        SimplifiedHanScript = 161,
        SinhalaScript = 162,
        SogdianScript = 163,
        SoraSompengScript = 164,
        SoyomboScript = 165,
        SundaneseScript = 166,
        SylotiNagriScript = 167,
        SymbolsScript = 168,
        SyriacScript = 169,
        TagalogScript = 170,
        TagbanwaScript = 171,
        TaiLeScript = 172,
        TaiVietScript = 173,
        TakriScript = 174,
        TamilScript = 175,
        TangsaScript = 176,
        TangutScript = 177,
        TeluguScript = 178,
        TengwarScript = 179,
        ThaanaScript = 180,
        ThaiScript = 181,
        TibetanScript = 182,
        TifinaghScript = 183,
        TirhutaScript = 184,
        TotoScript = 185,
        TraditionalHanScript = 186,
        UCASScript = 187,
        UgariticScript = 188,
        UnwrittenScript = 189,
        VaiScript = 190,
        VarangKshitiScript = 191,
        VisibleSpeechScript = 192,
        VithkuqiScript = 193,
        WanchoScript = 194,
        WesternSyriacScript = 195,
        WoleaiScript = 196,
        YezidiScript = 197,
        YiScript = 198,
        ZanabazarSquareScript = 199,
        ZawgyiScript = 200,

        HanifiScript = HanifiRohingyaScript,
        PersoArabicScript = ArabicScript,
        SimplifiedScript = SimplifiedHanScript,
        SumeroAkkadianCuneiformScript = SACuneiformScript,
        TraditionalScript = TraditionalHanScript,
        UnifiedCanadianAboriginalSyllabicsScript = UCASScript,

        LastScript = ZawgyiScript
    };
    // GENERATED PART ENDS HERE

    enum MeasurementSystem { MetricSystem, ImperialSystem, UKSystem };

    enum FormatType { LongFormat, ShortFormat, NarrowFormat };
    enum NumberOption {
        OmitGroupSeparator = 0x01,
        RejectGroupSeparator = 0x02
    };
    Q_DECLARE_FLAGS(NumberOptions, NumberOption)

    QLocale();
    QLocale(const QString &name);
    QLocale(Language language, Country country = AnyCountry);
    QLocale(Language language, Script script, Country country);
    QLocale(const QLocale &other);

    QLocale &operator=(const QLocale &other);

    Language language() const;
    Script script() const;
    Country country() const;
    QString name() const;

    QByteArray bcp47Name() const;

    short toShort(const QString &s, bool *ok = nullptr, int base = 0) const;
    ushort toUShort(const QString &s, bool *ok = nullptr, int base = 0) const;
    int toInt(const QString &s, bool *ok = nullptr, int base = 0) const;
    uint toUInt(const QString &s, bool *ok = nullptr, int base = 0) const;
    qlonglong toLongLong(const QString &s, bool *ok = nullptr, int base = 0) const;
    qulonglong toULongLong(const QString &s, bool *ok = nullptr, int base = 0) const;
    float toFloat(const QString &s, bool *ok = nullptr) const;
    double toDouble(const QString &s, bool *ok = nullptr) const;

    QString toString(qlonglong i) const;
    QString toString(qulonglong i) const;
    inline QString toString(short i) const;
    inline QString toString(ushort i) const;
    inline QString toString(int i) const;
    inline QString toString(uint i) const;
    QString toString(double i, char f = 'g', int prec = 6) const;
    inline QString toString(float i, char f = 'g', int prec = 6) const;
    QString toString(const QDate &date, const QString &formatStr) const;
    QString toString(const QDate &date, FormatType format = LongFormat) const;
    QString toString(const QTime &time, const QString &formatStr) const;
    QString toString(const QTime &time, FormatType format = LongFormat) const;
    QString toString(const QDateTime &dateTime, FormatType format = LongFormat) const;
    QString toString(const QDateTime &dateTime, const QString &format) const;

    QString dateFormat(FormatType format = LongFormat) const;
    QString timeFormat(FormatType format = LongFormat) const;
    QString dateTimeFormat(FormatType format = LongFormat) const;
#ifndef QT_NO_DATESTRING
    QDate toDate(const QString &string, FormatType = LongFormat) const;
    QTime toTime(const QString &string, FormatType = LongFormat) const;
    QDateTime toDateTime(const QString &string, FormatType format = LongFormat) const;
    QDate toDate(const QString &string, const QString &format) const;
    QTime toTime(const QString &string, const QString &format) const;
    QDateTime toDateTime(const QString &string, const QString &format) const;
#endif

    // ### Qt 5: We need to return QString from these function since
    //           unicode data contains several characters for these fields.
    QChar decimalPoint() const;
    QChar groupSeparator() const;
    QChar percent() const;
    QChar zeroDigit() const;
    QChar negativeSign() const;
    QChar positiveSign() const;
    QChar exponential() const;

    QString monthName(int, FormatType format = LongFormat) const;
    QString standaloneMonthName(int, FormatType format = LongFormat) const;
    QString dayName(int, FormatType format = LongFormat) const;
    QString standaloneDayName(int, FormatType format = LongFormat) const;

    Qt::DayOfWeek firstDayOfWeek() const;
    QList<Qt::DayOfWeek> weekdays() const;

    QString amText() const;
    QString pmText() const;

    MeasurementSystem measurementSystem() const;

    Qt::LayoutDirection textDirection() const;

    QString toUpper(const QString &str) const;
    QString toLower(const QString &str) const;

    inline bool operator==(const QLocale &other) const;
    inline bool operator!=(const QLocale &other) const;

    static QString languageToString(Language language);
    static QString countryToString(Country country);
    static QString scriptToString(Script script);

    static QLocale c() { return QLocale(C); }
    static QLocale system();

    static QList<QLocale> matchingLocales(QLocale::Language language, QLocale::Script script, QLocale::Country country);
    static QList<Country> countriesForLanguage(Language lang);

    void setNumberOptions(NumberOptions options);
    NumberOptions numberOptions() const;

private:
    friend class QLocalePrivate;
    // ### We now use this field to pack an index into locale_data and NumberOptions.
    // ### Qt 5: change to a QLocaleData *d; uint numberOptions.
    struct Data {
        quint16 index;
        QLocale::NumberOptions numberOptions;
    };

    Data p;
    const QLocalePrivate *d() const;
};
Q_DECLARE_TYPEINFO(QLocale, Q_MOVABLE_TYPE);
Q_DECLARE_OPERATORS_FOR_FLAGS(QLocale::NumberOptions)

inline QString QLocale::toString(short i) const
    { return toString(qlonglong(i)); }
inline QString QLocale::toString(ushort i) const
    { return toString(qulonglong(i)); }
inline QString QLocale::toString(int i) const
    { return toString(qlonglong(i)); }
inline QString QLocale::toString(uint i) const
    { return toString(qulonglong(i)); }
inline QString QLocale::toString(float i, char f, int prec) const
    { return toString(double(i), f, prec); }
inline bool QLocale::operator==(const QLocale &other) const
    { return d() == other.d() && numberOptions() == other.numberOptions(); }
inline bool QLocale::operator!=(const QLocale &other) const
    { return d() != other.d() || numberOptions() != other.numberOptions(); }

#ifndef QT_NO_DATASTREAM
Q_CORE_EXPORT QDataStream &operator<<(QDataStream &, const QLocale &);
Q_CORE_EXPORT QDataStream &operator>>(QDataStream &, QLocale &);
#endif

QT_END_NAMESPACE

#endif // QLOCALE_H
