/****************************************************************************
**
** Copyright (C) 2015 The Qt Company Ltd.
** Copyright (C) 2016 Ivailo Monev
**
** This file is part of the QtDBus module of the Katie Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
**
** GNU Lesser General Public License Usage
** This file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qdbusserver.h"
#include "qdbusconnection_p.h"
#include "qdbusconnectionmanager_p.h"


QT_BEGIN_NAMESPACE

/*!
    \class QDBusServer
    \inmodule QtDBus
    \internal

    \brief The QDBusServer class provides peer-to-peer communication
    between processes on the same computer.
*/

/*!
    Constructs a QDBusServer with the given \a address, and the given
    \a parent.
*/
QDBusServer::QDBusServer(const QString &address, QObject *parent)
    : QObject(parent)
{
    if (address.isEmpty())
        return;

    d = new QDBusConnectionPrivate(this);

    connect(d, SIGNAL(newServerConnection(QDBusConnection)),
                     this, SIGNAL(newConnection(QDBusConnection)));

    QDBusErrorInternal error;
    const QByteArray addressref = address.toUtf8();
    d->setServer(dbus_server_listen(addressref.constData(), error), error);
}

/*!
    Destructs a QDBusServer
*/
QDBusServer::~QDBusServer()
{
    if (QDBusConnectionManager::instance()) {
        QMutexLocker locker(&QDBusConnectionManager::instance()->mutex);
        Q_FOREACH (const QString &name, d->serverConnectionNames) {
            QDBusConnectionManager::instance()->removeConnection(name);
        }
        d->serverConnectionNames.clear();
        d->ref = 0;
        d->deleteLater();
    }
}

/*!
    Returns true if this QDBusServer object is connected.

    If it isn't connected, you need to call the constructor again.
*/
bool QDBusServer::isConnected() const
{
    return d && d->server && dbus_server_get_is_connected(d->server);
}

/*!
    Returns the last error that happened in this server.

    This function is provided for low-level code.
*/
QDBusError QDBusServer::lastError() const
{
    return d->lastError;
}

/*!
    Returns the address this server is associated with.
*/
QString QDBusServer::address() const
{
    QString addr;
    if (d && d->server) {
        char *c = dbus_server_get_address(d->server);
        addr = QString::fromUtf8(c);
        dbus_free(c);
    }

    return addr;
}

/*!
  \fn void QDBusServer::newConnection(const QDBusConnection &connection)

  This signal is emitted when a new client connection \a connection is
  established to the server.
 */

QT_END_NAMESPACE


#include "moc_qdbusserver.h"
