#include "main_activity.h"
#include "../settings.h"

#include <unistd.h>
#ifdef __FreeBSD__
#include <sys/sysctl.h>
#include <sys/user.h>
#include <sys/vmmeter.h>
#elif defined(__APPLE__)
#include <sys/sysctl.h>
#include <mach/mach.h>
#else
#include <sys/sysinfo.h>
#endif
#include <file_picker_factory.h>
#include <game_window_manager.h>
#include "uuid.h"
#include <climits>
#include <sstream>
#include <android/keycodes.h>

#include <log.h>

FakeJni::JInt BuildVersion::SDK_INT = 32;
std::shared_ptr<FakeJni::JString> BuildVersion::RELEASE = std::make_shared<FakeJni::JString>("AndroidX");

std::shared_ptr<FakeJni::JString> MainActivity::createUUID() {
    return UUID::randomUUID()->toString();
}

FakeJni::JFloat MainActivity::getPixelsPerMillimeter() {
    // assume 96 DPI for now with GUI scale of 2
    return (96 / 25.4f) * 2 * Settings::scale;
}

FakeJni::JInt MainActivity::getPlatformDpi() {
    // assume 96 DPI for now with GUI scale of 2
    return 96 * 2 * Settings::scale;
}

FakeJni::JLong MainActivity::getUsedMemory() {
#ifdef __APPLE__
    uint64_t page_size;
    size_t len = sizeof(page_size);
    sysctlbyname("hw.pagesize", &page_size, &len, NULL, 0);

    struct vm_statistics64 stat;
    mach_msg_type_number_t count = HOST_VM_INFO64_COUNT;
    host_statistics64(mach_host_self(), HOST_VM_INFO64, (host_info64_t)&stat, &count);

    double page_K = page_size / (double)1024;
    return (stat.active_count + stat.wire_count) * page_K * 1000;
#elif defined(__FreeBSD__)
    int mib[4] = {CTL_VM, KERN_PROC, KERN_PROC_PID, getpid()};
    struct kinfo_proc info = {};
    size_t size = sizeof(info);
    if(sysctl(mib, 4, &info, &size, NULL, 0) != 0)
        return 0L;
    // from usr.bin/top/machine.c macro PROCSIZE
    return info.ki_size / 1024;
#else
    FILE* file = fopen("/proc/self/statm", "r");
    if(file == nullptr)
        return 0L;
    int pageSize = getpagesize();
    long long pageCount = 0L;
    fscanf(file, "%lld", &pageCount);
    fclose(file);
    return pageCount * pageSize;
#endif
}

FakeJni::JLong MainActivity::getFreeMemory() {
#ifdef __APPLE__
    uint64_t page_size;
    size_t len = sizeof(page_size);
    sysctlbyname("hw.pagesize", &page_size, &len, NULL, 0);

    struct vm_statistics64 stat;
    mach_msg_type_number_t count = HOST_VM_INFO64_COUNT;
    host_statistics64(mach_host_self(), HOST_VM_INFO64, (host_info64_t)&stat, &count);

    double page_K = page_size / (double)1024;
    return stat.free_count * page_K * 1000;
#elif defined(__FreeBSD__)
    uint64_t page_size;
    size_t len = sizeof(page_size);
    sysctlbyname("hw.pagesize", &page_size, &len, NULL, 0);

    int mib[2] = {CTL_VM, VM_TOTAL};
    struct vmtotal info = {};
    size_t size = sizeof(info);
    if(sysctl(mib, 2, &info, &size, NULL, 0) != 0)
        return 0L;

    double page_K = page_size / (double)1024;
    return info.t_free * page_K * 1000;
#else
    struct sysinfo memInfo;
    sysinfo(&memInfo);
    long long total = memInfo.freeram;
    total *= memInfo.mem_unit;
    return total;
#endif
}

FakeJni::JLong MainActivity::getTotalMemory() {
#ifdef __APPLE__
    uint64_t memsize;
    size_t len = sizeof(memsize);
    sysctlbyname("hw.memsize", &memsize, &len, NULL, 0);
    return memsize;
#elif defined(__FreeBSD__)
    uint64_t page_size;
    size_t len = sizeof(page_size);
    sysctlbyname("hw.pagesize", &page_size, &len, NULL, 0);

    int mib[2] = {CTL_VM, VM_TOTAL};
    struct vmtotal info = {};
    size_t size = sizeof(info);
    if(sysctl(mib, 2, &info, &size, NULL, 0) != 0)
        return 0L;

    double page_K = page_size / (double)1024;
    return info.t_vm * page_K * 1000;
#else
    struct sysinfo memInfo;
    sysinfo(&memInfo);
    long long total = memInfo.totalram;
    total *= memInfo.mem_unit;
    return total;
#endif
}

FakeJni::JLong MainActivity::getMemoryLimit() {
    return getTotalMemory();
}

FakeJni::JLong MainActivity::getAvailableMemory() {
    return getTotalMemory();
}

void MainActivity::pickImage(FakeJni::JLong callback) {
    try {
        auto picker = FilePickerFactory::createFilePicker();
        picker->setTitle("Select image");
        picker->setFileNameFilters({"*.png"});
        if(picker->show()) {
            auto method = getClass().getMethod("(JLjava/lang/String;)V", "nativeOnPickImageSuccess");
            FakeJni::LocalFrame frame;
            method->invoke(frame.getJniEnv(), this, callback, frame.getJniEnv().createLocalReference(std::make_shared<FakeJni::JString>(picker->getPickedFile())));
        } else {
            auto method = getClass().getMethod("(J)V", "nativeOnPickImageCanceled");
            FakeJni::LocalFrame frame;
            method->invoke(frame.getJniEnv(), this, callback);
        }
    } catch(const std::exception& e) {
        GameWindowManager::getManager()->getErrorHandler()->onError("FilePickerFactory", std::string("Failed to open the file-picker details: ") + e.what());
        auto method = getClass().getMethod("(J)V", "nativeOnPickImageCanceled");
        FakeJni::LocalFrame frame;
        method->invoke(frame.getJniEnv(), this, callback);
    }
}

void MainActivity::initializeXboxLive(FakeJni::JLong xalinit, FakeJni::JLong xblinit) {
    auto method = getClass().getMethod("(JJ)V", "nativeInitializeXboxLive");
    FakeJni::LocalFrame frame;
    method->invoke(frame.getJniEnv(), this, xalinit, xblinit);
}

void MainActivity::requestIntegrityToken(std::shared_ptr<FakeJni::JString> str) {
    auto method = getClass().getMethod("(Ljava/lang/String;)V", "nativeSetIntegrityToken");
    FakeJni::LocalFrame frame;
    method->invoke(frame.getJniEnv(), this, createUUID());
}

void MainActivity::launchUri(std::shared_ptr<FakeJni::JString> url) {
    int pid;
    if ((pid = fork())) {
    } else {
    #ifdef __APPLE__
        execl("/usr/bin/open", "/usr/bin/open", url->asStdString().c_str(), NULL);
    #else
        execl("/usr/bin/xdg-open", "/usr/bin/xdg-open", url->asStdString().c_str(), NULL);
    #endif
        _Exit(0);
    }
}

void MainActivity::setClipboard(std::shared_ptr<FakeJni::JString> tocopy) {
    window->setClipboardText(tocopy->asStdString());
}

void MainActivity::share(std::shared_ptr<FakeJni::JString> title, std::shared_ptr<FakeJni::JString> string, std::shared_ptr<FakeJni::JString> url) {
    if ((title->asStdString().find("\"") == std::string::npos) && (title->asStdString().find("\\") == std::string::npos) && (string->asStdString().find("\"") == std::string::npos) && (string->asStdString().find("\\") == std::string::npos)) {
        int pid;
        if ((pid = fork())) {
        } else {
        #ifdef __APPLE__
            execl("/usr/bin/osascript", "/usr/bin/osascript", "-e", ("display alert \"" + title->asStdString() + "\" message \"" + string->asStdString() + "\n" + url->asStdString() + "\"").c_str(), NULL);
        #else
            execl("/usr/bin/zenity", "/usr/bin/zenity", "--info", "--title", title->asStdString().c_str(), "--text", (string->asStdString() + "\n" + url->asStdString()).c_str(), NULL);
        #endif
            _Exit(0);
        }
    }
}

FakeJni::JInt MainActivity::getCaretPosition() {
    ignoreNextHideKeyboard = false;
    if(textInput) {
        return textInput->getCursorPosition();
    }
    return -1;
}

void MainActivity::setCaretPosition(FakeJni::JInt pos) {
    if(textInput)
        textInput->setCursorPosition(pos);
}

FakeJni::JLong MainActivity::initializeXboxLive2(FakeJni::JLong xalinit, FakeJni::JLong xblinit) {
    auto method = getClass().getMethod("(JJ)V", "nativeInitializeXboxLive");
    FakeJni::LocalFrame frame;
    auto ret = method->invoke(frame.getJniEnv(), this, xalinit, xblinit);
    return ret.j;
}

FakeJni::JLong MainActivity::initializeLibHttpClient(FakeJni::JLong init) {
    auto method = getClass().getMethod("(J)J", "nativeinitializeLibHttpClient");
    if(!method) {
        method = getClass().getMethod("(J)J", "nativeInitializeLibHttpClient");
    }
    FakeJni::LocalFrame frame;
    auto ret = method->invoke(frame.getJniEnv(), this, init);
    return ret.j;
}

std::shared_ptr<FakeJni::JIntArray> MainActivity::getImageData(std::shared_ptr<FakeJni::JString> filename) {
    if(!stbi_load_from_memory || !stbi_image_free)
        return 0;
    int width, height, channels;
    std::ifstream f(filename->asStdString().data());
    if(!f.is_open())
        return 0;
    std::stringstream s;
    s << f.rdbuf();
    auto buf = s.str();
    auto image = stbi_load_from_memory((unsigned char*)buf.data(), buf.length(), &width, &height, &channels, 4);
    if(!image)
        return 0;
    auto ret = std::make_shared<FakeJni::JIntArray>(2 + width * height);
    (*ret)[0] = width;
    (*ret)[1] = height;

    for(int x = 0; x < width * height; x++) {
        (*ret)[2 + x] = (image[x * 4 + 2]) | (image[x * 4 + 1] << 8) | (image[x * 4 + 0] << 16) | (image[x * 4 + 3] << 24);
    }
    stbi_image_free(image);
    return ret;
}

std::shared_ptr<FakeJni::JByteArray> MainActivity::getFileDataBytes(std::shared_ptr<FakeJni::JString> path) {
    return std::make_shared<FakeJni::JByteArray>();
}

// Allow Marketplace Content bigger than 432MB to download
FakeJni::JBoolean MainActivity::supportsSizeQuery(std::shared_ptr<FakeJni::JString> path) {
    // const char* rpath = path->data();
    // printf("supportsSizeQuery: %s\n", rpath);
    return true;
}

FakeJni::JLong MainActivity::getAllocatableBytes(std::shared_ptr<FakeJni::JString> path) {
    // const char* rpath = path->data();
    // printf("getAllocatableBytes: %s\n", rpath);
    return 1024LL * 1024LL * 1024LL * 1024LL;
}

void MainActivity::startPlayIntegrityCheck() {
    // auto method = PlayIntegrity::getDescriptor()->getMethod("()V", "nativePlayIntegrityComplete");
    // FakeJni::LocalFrame frame;
    // method->invoke(frame.getJniEnv(), PlayIntegrity::getDescriptor().get());
}

void MainActivity::openFile() {
    try {
        auto picker = FilePickerFactory::createFilePicker();
        picker->setTitle("Select file");
        if(picker->show()) {
            auto method = getClass().getMethod("(Ljava/lang/String;)V", "nativeOnPickFileSuccess");
            FakeJni::LocalFrame frame;
            method->invoke(frame.getJniEnv(), this, frame.getJniEnv().createLocalReference(std::make_shared<FakeJni::JString>(picker->getPickedFile())));
        } else {
            auto method = getClass().getMethod("()V", "nativeOnPickFileCanceled");
            FakeJni::LocalFrame frame;
            method->invoke(frame.getJniEnv(), this);
        }
    } catch(const std::exception& e) {
        GameWindowManager::getManager()->getErrorHandler()->onError("FilePickerFactory", std::string("Failed to open the file-picker details: ") + e.what());
        auto method = getClass().getMethod("()V", "nativeOnPickFileCanceled");
        FakeJni::LocalFrame frame;
        method->invoke(frame.getJniEnv(), this);
    }
}
void MainActivity::saveFile(std::shared_ptr<FakeJni::JString> fileName) {
    try {
        auto picker = FilePickerFactory::createFilePicker();
        picker->setMode(FilePicker::Mode::SAVE);
        picker->setTitle("Select file");
        picker->setFileName(fileName->asStdString());
        if(picker->show()) {
            auto method = getClass().getMethod("(Ljava/lang/String;)V", "nativeOnPickFileSuccess");
            FakeJni::LocalFrame frame;
            method->invoke(frame.getJniEnv(), this, frame.getJniEnv().createLocalReference(std::make_shared<FakeJni::JString>(picker->getPickedFile())));
        } else {
            auto method = getClass().getMethod("()V", "nativeOnPickFileCanceled");
            FakeJni::LocalFrame frame;
            method->invoke(frame.getJniEnv(), this);
        }
    } catch(const std::exception& e) {
        GameWindowManager::getManager()->getErrorHandler()->onError("FilePickerFactory", std::string("Failed to open the file-picker details: ") + e.what());
        auto method = getClass().getMethod("()V", "nativeOnPickFileCanceled");
        FakeJni::LocalFrame frame;
        method->invoke(frame.getJniEnv(), this);
    }
}

FakeJni::JInt MainActivity::getKeyFromKeyCode(FakeJni::JInt keyCode, FakeJni::JInt metaState, FakeJni::JInt deviceId) {
    if(!Settings::enable_keyboard_autofocus_patches_1_20_60) {
        return 0;
    }
    if(keyCode >= AKEYCODE_F1 && keyCode <= AKEYCODE_F12) {
        return 0;
    }
    auto ret = lastChar;
    switch (keyCode)
    {
    case AKEYCODE_DEL:
    case AKEYCODE_FORWARD_DEL:
    case AKEYCODE_SHIFT_LEFT:
    case AKEYCODE_SHIFT_RIGHT:
    case AKEYCODE_ALT_LEFT:
    case AKEYCODE_ALT_RIGHT:
    case AKEYCODE_CTRL_LEFT:
    case AKEYCODE_CTRL_RIGHT:
    case AKEYCODE_CAPS_LOCK:
    case AKEYCODE_META_LEFT:
    case AKEYCODE_META_RIGHT:
    case AKEYCODE_ESCAPE:
    case AKEYCODE_ENTER:
    case AKEYCODE_VOLUME_UP:
    case AKEYCODE_VOLUME_DOWN:
    case AKEYCODE_VOLUME_MUTE:
    case AKEYCODE_DPAD_LEFT:
    case AKEYCODE_DPAD_RIGHT:
    case AKEYCODE_DPAD_UP:
    case AKEYCODE_DPAD_UP_LEFT:
    case AKEYCODE_DPAD_UP_RIGHT:
    case AKEYCODE_DPAD_DOWN:
    case AKEYCODE_DPAD_DOWN_LEFT:
    case AKEYCODE_DPAD_DOWN_RIGHT:
    case AKEYCODE_UNKNOWN:
        return 0;
    }
    lastChar = 0;
    return ret;
}

void MainActivity::setLastChar(FakeJni::JInt sym) {
    lastChar = sym;
}
